#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "vsmpg.h"
#include "loud.h"


/*

  loud.c - Loudness main program 2003-01-16

  This is the main program for loudness. Depending on the filter use it
  calls either Filter() or Filter2(). Filter() has a second degree FIR filter
  for high freqeuncy filtering followed by two first degree IIR filters for
  low frequency filtering. Filter2() has a first defree FIR filter followed
  by one first degree IIR filter. While Filter() is more powerful, Filter2()
  has less noise, is more stable, and it is also slightly faster.

 */






#ifdef __VSDSP__
struct FILT {
  s_int16 fir[3];	/* Index 0 */
  s_int16 *findex;	/* Index 3 */
  long lp[2];		/* Index 4,5,6,7 */
};

extern struct FILT stereoFilt[2];
#else
struct FILT {
  s_int16 fir[2*3];	/* Index 0 */
  s_int16 findex;	/* Index 6 */
  long lp[2];		/* Index 7 */
			/* Index 11 */
};

struct FILT stereoFilt[2] = {{{0}}};
#endif


#ifdef __VSDSP__
extern __y s_int16 consts[5];
#else
__y s_int16 consts[5];
long lp0max, lp1max, inmax;
#endif

s_int16 useF2 = 0;




auto void Filter(register __i0 s_int16 *in, register __i1 s_int16 step,
		 register __a0 s_int16 n, register __i2 struct FILT *filt);
auto void Filter2(register __i0 s_int16 *in, register __i1 s_int16 step,
		  register __a0 s_int16 n, register __i2 struct FILT *filt);
void InitFilters(register s_int16 n);

s_int16 LoudMain(register s_int16 __i0 **data, register s_int16 __a0 nSampl,
	       register __a1 s_int16 mode) {
  if (mode == APPL_RESET) {
    InitFilters(0);
  } else if (mode == APPL_W0) {
    InitFilters(nSampl);
  } else if (mode == APPL_AUDIO) {
    if (useF2 > 0) {
      Filter2(*data, 2, nSampl, &stereoFilt[0]);
      Filter2(*data+1, 2, nSampl, &stereoFilt[1]);
    } else if (!useF2) {
      Filter(*data, 2, nSampl, &stereoFilt[0]);
      Filter(*data+1, 2, nSampl, &stereoFilt[1]);
    }
  }
  return nSampl;
}




void InitFilters(register s_int16 n) {
  register int i;
  char *t = (char *)stereoFilt;
  if (!n) {
    for (i=0; i<sizeof(stereoFilt); i++)
      *t++ = 0;
#ifdef __VSDSP__
    stereoFilt[0].findex = &stereoFilt[0].fir[0];
    stereoFilt[1].findex = &stereoFilt[1].fir[0];
#else
    inmax = lp0max = lp1max = 0;
#endif
  } else {
#ifdef __VSDSP__OR_0
    /*
      If VS1001 has for some reason been software reset, check if the
      FIR pointers have been destroyed and restore them.
    */
    if (!stereoFilt[0].findex || !stereoFilt[1].findex) {
      stereoFilt[0].findex = &stereoFilt[0].fir[0];
      stereoFilt[1].findex = &stereoFilt[1].fir[0];
    }
#endif
    if (n == 1) {
      consts[0] =  17034;	/*  2.25   = b1[1] */
      consts[1] =  15856;	/*  2.09   = b1[3] */
      consts[2] = -32768;	/* -4.32   = b1[2] */
      consts[3] =   8064;	/* -0.9844 = a1[2] */
    } else if (n == 2) {
      consts[0] =  14917;	/*  2.4592 = b1[1] */
      consts[1] =  18013;	/*  2.0365 = b1[3] */
      consts[2] = -32768;	/* -4.4792 = b1[2] */
      consts[3] =   8028;	/* -0.9800 = a1[2] */
    } else if (n == 3) {
      consts[0] =  17817;	/*         = b1[1] */
      consts[1] =  15081;	/*         = b1[3] */
      consts[2] = -32768;	/*         = b1[2] */
      consts[3] =   7946;	/*         = a1[2] */
    } else if (n == 4) {
      consts[0] =  18514;	/*         = b1[1] */
      consts[1] =  14500;	/*         = b1[3] */
      consts[2] = -32768;	/*         = b1[2] */
      consts[3] =   8069;	/*         = a1[2] */
    } else if (n == 5) {
      consts[0] =  16384;	/*         = firb1 */
      consts[1] =      0;	/*         = firb2 */
      consts[2] =  -8028;	/*         = iirb  */
      consts[3] =  -8126;	/*         = iira  */
    } else if (n == 6) {
      consts[0] =  24576;	/*         = firb1 */
      consts[1] =  -6160;	/*         = firb2 */
      consts[2] =  -8028;	/*         = iirb  */
      consts[3] =  -8126;	/*         = iira  */
    } else if (n == 7) {
      consts[0] =  21856;	/*         = firb1 */
      consts[1] =  -7645;	/*         = firb2 */
      consts[2] =  -7987;	/*         = iirb  */
      consts[3] =  -8126;	/*         = iira  */
    } else if (n == 8) {
      consts[0] =  26444;	/*         = firb1 */
      consts[1] =   6324;	/*         = firb2 */
      consts[2] =  -8159;	/*         = iirb  */
      consts[3] =  -8126;	/*         = iira  */
    } else if (n == 9) {
      consts[0] =  16384;	/*         = firb1 */
      consts[1] =      0;	/*         = firb2 */
      consts[2] =  -7782;	/*         = iirb  */
      consts[3] =  -8028;	/*         = iira  */
    } else if (n == 10) {
      consts[0] =  24576;	/*         = firb1 */
      consts[1] =  -6160;	/*         = firb2 */
      consts[2] =  -7700;	/*         = iirb  */
      consts[3] =  -7987;	/*         = iira  */
    } else if (n == 11) {
      consts[0] =  21856;	/*         = firb1 */
      consts[1] =  -7645;	/*         = firb2 */
      consts[2] =  -7782;	/*         = iirb  */
      consts[3] =  -8045;	/*         = iira  */
    } else if (n == 12) {
      consts[0] =  16384;	/*         = firb1 */
      consts[1] =  16384;	/*         = firb2 */
      consts[2] =  -8110;	/*         = iirb  */
      consts[3] =  -7864;	/*         = iira  */
    }
    consts[4] = 0;
    useF2 = (n > 4);
  }
}

#ifdef __VSDSP__
auto void Filter(register __i0 s_int16 *in, register __i1 s_int16 step,
		 register __a0 s_int16 n, register __i2 struct FILT *filt);
#else
auto void Filter(register __i0 s_int16 *in, register __i1 s_int16 step,
		 register __a0 s_int16 n, register __i2 struct FILT *filt) {
  int i;
  for (i=0; i<n; i++) {
    register long f = 0;
    register short *dl;

    dl = &(filt->fir[filt->findex]);
    *dl = *in;
    f += *dl++ * consts[0];
    f += *dl++ * consts[1];
    f += *dl++ * consts[2];
    *dl = *in;

    if (++filt->findex > 2)
      filt->findex = 0;

#ifndef __VSDSP__
    if (f > 0 && inmax < f)
      inmax = f;
    if (f < 0 && inmax < -f)
      inmax = -f;
#endif

    filt->lp[0] = (f + consts[3]*filt->lp[0]) >> 13;
    filt->lp[1] = (filt->lp[0] + filt->lp[1] / 2);
    f = filt->lp[1] / 2;

#ifndef __VSDSP__
    if (filt->lp[0] > 0 && lp0max < filt->lp[0])
      lp0max = filt->lp[0];
    if (filt->lp[0] < 0 && lp0max < -filt->lp[0])
      lp0max = -filt->lp[0];
    if (filt->lp[1] > 0 && lp1max < filt->lp[1])
      lp1max = filt->lp[1];
    if (filt->lp[1] < 0 && lp1max < -filt->lp[1])
      lp1max = -filt->lp[1];
#endif

#if 0
    fprintf(stderr, "f %ld lp0 %ld lp1 %ld\n", f, filt->lp[0], filt->lp[1]);
#endif
    if (f >= 32767)
      *in = 32767;
    else if (f <= -32768)
      *in = -32768;
    else
      *in = (s_int16)f;

    in += step;
  }
}
#endif

#ifdef __VSDSP__
auto void Filter2(register __i0 s_int16 *in, register __i1 s_int16 step,
		  register __a0 s_int16 n, register __i2 struct FILT *filt);
#else
auto void Filter2(register __i0 s_int16 *in, register __i1 s_int16 step,
		  register __a0 s_int16 n, register __i2 struct FILT *filt) {
  int i;
  for (i=0; i<n; i++) {
    register long f = 0;
    f = ((consts[0] * (s_int32)*in) + (consts[1] * (s_int32)filt->fir[0]))
      >> 15;
    filt->fir[0] = *in;

    {
      long t = (f + filt->lp[0]);
      filt->lp[0] = (consts[2]*f - consts[3]*t) >> 13;

      if (t >= 32767)
	*in = 32767;
      else if (t <= -32768)
	*in = -32768;
      else
	*in = (s_int16)t;
    }

    in += step;
  }
}
#endif
