// define USE_MOJIBAKE to call character decoder library
#define USE_MOJIBAKE
//
// Audio Decoder Model
//
#include <string.h>
#include <timers.h>
#include <libaudiodec.h>
#include <apploader.h>
#include <uimessages.h>
#include <abstract_playlist.h>
#include <vsostasks.h>
#include <volink.h>
#include "mutex.h"

static u_int16 mutex=0;

void ModelTask(void);
ioresult ModelCallback(s_int16 index,u_int16 message,u_int32 value);

extern void* modelCallbacks[];

void init() {
	modelCallbacks[0]=ModelCallback; // temporary; publish our model callback
	InitMutexN(&mutex,1);
	StartTask(TASK_DECODER,ModelTask); // in mp3model.c
	Yield(); // Release the CPU to allow the MP3 decoder task to initialize
}

void fini() {
	// should probably do something here
	modelCallbacks[0]=NULL;
}
//
// Signals
//
#define S_JUMP_CURRENT		(1<<0)
#define S_JUMP_NEXT			(1<<1)
#define S_JUMP_PREV			(1<<2)
#define S_JUMP_FIRST			(1<<3)
#define S_JUMP_LAST			(1<<4)
#define S_POSITION_SAVE		(1<<5)
#define S_POSITION_RESTORE	(1<<6)
#define S_SEND_MESSAGE_LIST	(1<<7)
#define S_CLOSE_DECODER		(1<<8)
#define S_IDLECHECK			(1<<15)

typedef unsigned int SIGNALS;

#define SFP_FILENAME_SIZE 256
//
// single file playlist is used for serving UIMSG_TEXT_OPEN_FILE requests
//
typedef struct SINGLE_FILE_PLAYLIST {
	ABSTRACT_PLAYLIST a;
	char name[SFP_FILENAME_SIZE];
} SINGLE_FILE_PLAYLIST;
//
extern SINGLE_FILE_PLAYLIST singleFilePlaylist; // forward declaration
//
// local state variables
//
static SIGNALS signal=0; // ModelCallback --> ModelTask signaling
static AUDIO_DECODER* auDec=NULL; // currently open audio decoder
static void* audioDecLib=NULL; // library which contains the decoder
static ABSTRACT_PLAYLIST* playlist=(ABSTRACT_PLAYLIST*)&singleFilePlaylist;
static s_int16 currentIndex=0; // index of currently selected playlist item
static s_int16 lastIndex=0; // length of the current playlist
static u_int32 lastSecond=-1;
static u_int16 lastPause=-1;
static s_int16 jumpDirection=1; // direction of the playlist autoadvance jump
static s_int16 jumpTo=-1;
// this is the callback function we call when we need to communicate with the outside world
static UICallback playerCallback=NULL;
//
#define MAX_TMP_STR_LEN 256 /* Must be at least 158 words, requirement by DecodeID3 */
static u_int16 currTmpStr[MAX_TMP_STR_LEN];

static void SendMsg(register s_int16 index,register u_int16 message,register u_int32 value) {
	if(playerCallback) {
		playerCallback(index,message,value);
	}
}

//
void DecodeID3(register FILE* fp,register UICallback);
void QueryPlaylist(register s_int16 index,register UICallback);
//
// The list of UI messages we claim to handle. This list provides a hint for the UI module
// about the capabilities of this model.
//
static UiMessageListItem __y playerUiMessageListItem[]={
	{UIMSG_VSOS_SET_CALLBACK_FUNCTION,"Callback"},
	{UIMSG_VSOS_SET_CURRENT_PLAYLIST,"Playlist"},
	//
	{UIMSG_INPUTS,"File Controls"}, // Start of input category "File Controls"
	//
	{UIMSG_BUT_FIRST,"First"},
	{UIMSG_BUT_PREVIOUS,"Previous"},
	{UIMSG_BUT_NEXT,"Next"},
	{UIMSG_BUT_LAST,"Last"},
	{UIMSG_TEXT_OPEN_FILE,"Open File"},
	//
	{UIMSG_INPUTS,"Play Controls"}, // Start of input category "Play Controls"
	//
	{UIMSG_BUT_PLAY,"Play"},
	{UIMSG_BUT_RESTART,"Restart"},
	{UIMSG_BUT_STOP,"Stop"},
	{UIMSG_BUT_PAUSE,"Pause"},
	{UIMSG_BUT_PAUSE_TOGGLE,"Pause Toggle"},
	{UIMSG_BOOL_FAST_FORWARD,"Fast Forward"},
	{UIMSG_BUT_CLOSE_DECODER,"Close decoder"},
/*	{UIMSG_BUT_SAVE_POS,"Save position"},
	{UIMSG_BUT_RESTORE_POS,"Restore position"},*/
	//
	{UIMSG_OUTPUTS,"Song Info"}, // Start of output category "Song Info"
	//
	{UIMSG_TEXT_LONG_FILE_NAME,"Now Playing"},
	{UIMSG_TEXT_ARTIST_NAME,"Artist"},
	{UIMSG_TEXT_SONG_NAME,"Song"},
	{UIMSG_TEXT_ALBUM_NAME,"Album"},
	{UIMSG_TEXT_YEAR,"Year"},
	{UIMSG_U32_PLAY_FILE_PERCENT,"Play Time"},
	{UIMSG_U32_PLAY_TIME_SECONDS,""},
	{UIMSG_VSOS_END_OF_LIST,NULL}
};

static void SetJumpFlag(register SIGNALS flag) {
	signal&=~(S_JUMP_CURRENT|S_JUMP_NEXT|S_JUMP_PREV|S_JUMP_FIRST|S_JUMP_LAST);
	signal|=flag;
}

/*
  This callback function gets called by the MP3 player View / Controller
  part (UI).

  Because the standard Decoder is built in such a way that Decode() functions
  doesn't end execution, we have to insert a Cancel command for all the
  user operations that require action from our MP3 model while doing playback.
  Although it makes the state machine operations less clear, we need to
  put some of the state machine's operations here.
*/

DLLENTRY(ModelCallback)

ioresult ModelCallback(s_int16 index,u_int16 message,u_int32 value) {
	struct Codec* codec;
	s_int16 shouldCancel;
	s_int16 jumpFlag;
	Forbid();
	codec=(auDec)?auDec->cod:NULL;
	jumpFlag=-1;
	shouldCancel=0;
	switch(message) {
	case UIMSG_VSOS_SET_CALLBACK_FUNCTION:
		playerCallback=(UICallback)value;
		break;
	case UIMSG_VSOS_SET_CURRENT_PLAYLIST:
		playlist=(ABSTRACT_PLAYLIST*)value;
		lastIndex=playlist?playlist->Size(playlist)-1:0;
		break;
	case UIMSG_VSOS_GET_MSG_LIST:
		signal=S_SEND_MESSAGE_LIST;
		break;
	case UIMSG_TEXT_OPEN_FILE:
		playlist=(ABSTRACT_PLAYLIST*)&singleFilePlaylist;
		strcpy(singleFilePlaylist.name,(char*)value);
		lastIndex=0;
		jumpFlag=S_JUMP_FIRST;
		break;
	case UIMSG_BUT_FIRST:
		shouldCancel=1;	
		jumpFlag=S_JUMP_FIRST;
		break;
	case UIMSG_BUT_NEXT:
		shouldCancel=1;
		jumpFlag=S_JUMP_NEXT;
		break;
	case UIMSG_BUT_PREVIOUS:
		shouldCancel=1;
		jumpFlag=S_JUMP_PREV;
		break;
	case UIMSG_BUT_LAST:
		shouldCancel=1;
		jumpFlag=S_JUMP_LAST;
		break;
	case UIMSG_BUT_RESTART:
		shouldCancel=1;
		jumpFlag=S_JUMP_CURRENT;
		break;
	case UIMSG_BUT_STOP:
		shouldCancel=1;
		jumpFlag=0;
		break;
	case UIMSG_BUT_PLAY:
		if(auDec) {
			auDec->pause=0;
		} else {
			jumpFlag=S_JUMP_CURRENT;
		}
		break;
	case UIMSG_BUT_PAUSE:
		if(auDec) {
			auDec->pause=(s_int16)value;
		}
		break;
	case UIMSG_BUT_PAUSE_TOGGLE:
		if(auDec) {
			auDec->pause=!(auDec->pause);
		}
		break;
	case UIMSG_BOOL_FAST_FORWARD:
		if(codec) {
			codec->cs->fastForward=(value?8:1);
		}
		break;
	case UIMSG_BUT_CLOSE_DECODER:
		signal|=S_CLOSE_DECODER;
		break;
/*	case UIMSG_BUT_SAVE_POS:
		signal|=S_POSITION_SAVE;
		break;
	case UIMSG_BUT_RESTORE_POS:
		signal|=S_POSITION_RESTORE;
		break;*/
	case UIMSG_S16_SONG_NUMBER:
		shouldCancel=1;
		jumpFlag=S_JUMP_CURRENT;
		jumpTo=(s_int16)value;
		break;		
	case UIMSG_VSOS_QUERY_PLAYLIST_ITEM:
		ObtainMutex(&mutex);
		QueryPlaylist(index,(UICallback)value);
		ReleaseMutex(&mutex);
		break;
	default:
		break;
	}
	if(jumpFlag>=0) {
		SetJumpFlag(jumpFlag);
		jumpDirection=(jumpFlag==S_JUMP_PREV)?-1:1;
		if(auDec) {
			auDec->pause=0;
		}
	}
	if(shouldCancel && codec) {
		codec->cs->cancel=1;
	}
	Permit();
}

void PlayAudioFile(); // forward declaration

//
// ModelTask responds to signals received via ModelCallback
//
void ModelTask(void) {
	audioDecLib=LoadLibrary("AUDIODEC");
	if(audioDecLib) {
		while(!(signal&S_CLOSE_DECODER)) {
			if(signal&S_SEND_MESSAGE_LIST) {
				SendMsg(-1,UIMSG_VSOS_MSG_LIST,(u_int32)playerUiMessageListItem);
			} else if(signal&S_JUMP_FIRST) {
				currentIndex=0;			
				SetJumpFlag(S_JUMP_CURRENT);
			} else if(signal&S_JUMP_PREV) {
				--currentIndex;
				SetJumpFlag(S_JUMP_CURRENT);
			} else if(signal&S_JUMP_LAST) {
				currentIndex=lastIndex;			
				SetJumpFlag(S_JUMP_CURRENT);
			} else if(signal&S_JUMP_NEXT) {
				++currentIndex;
				SetJumpFlag(S_JUMP_CURRENT);
			} else if(signal&S_JUMP_CURRENT) {
				if(jumpTo>=0) {
					currentIndex=jumpTo;
					jumpTo=-1;
				}
				SetJumpFlag(0);
				PlayAudioFile();
			} else if(signal&S_IDLECHECK) {
				signal&=~S_IDLECHECK;			
				if(playlist->flags&PF_AUTOADVANCE) {
					currentIndex+=jumpDirection;
					SetJumpFlag(S_JUMP_CURRENT);
				}
			} else {
				Delay(10);
			}
		}
		DropLibrary(audioDecLib);
		audioDecLib=NULL;
	}
	SendMsg(-1,UIMSG_BUT_DECODER_CLOSED,1);
}

void CodecServicesCallback(AUDIO_DECODER*,u_int16); // forward declaration

static void PlayAudioFile() {
	FILE* inFp=0;
	u_int16 result=ceOk;
	u_int16 flags=playlist->flags;
	ObtainMutex(&mutex);
	if(flags&PF_SIZE_CHANGED) {
		playlist->flags&=~PF_SIZE_CHANGED;
		lastIndex=playlist->Size(playlist);
	}
	if(currentIndex>lastIndex) {
		if(flags&PF_LOOPING) {
			currentIndex=0;
		} else {
			currentIndex=lastIndex;
			SendMsg(-1,UIMSG_VSOS_NO_MORE_FILES,0);
			ReleaseMutex(&mutex);
			return;
		}
	} else if(currentIndex<0) {
		currentIndex=0;
		SendMsg(-1,UIMSG_VSOS_NO_MORE_FILES,0);
		ReleaseMutex(&mutex);
		return;
	}
	inFp=playlist->Open(playlist,currentIndex);
	if(inFp) {
		auDec=CreateAudioDecoder(audioDecLib,inFp,stdaudioout,CodecServicesCallback,auDecFGuess);
		if(auDec) {
			const char* eStr;
			// we have a decoder for the file, let's do this thing
			// ask the playlist to identify the current item
			playlist->Name(playlist,currentIndex,(wchar*)currTmpStr,sizeof(currTmpStr));
			if(playerCallback && currTmpStr[0]) { // got response, so send it forward
				playerCallback(-1,UIMSG_TEXT_LONG_FILE_NAME,(u_int32)currTmpStr);
			}
			if((auDec->cs.fileLeft!=0xFFFFFFFFU) && playerCallback) {
				// parse tags
				DecodeID3(inFp,playerCallback);
			}
			ReleaseMutex(&mutex);
			result=DecodeAudio(audioDecLib,auDec,&eStr);
			ObtainMutex(&mutex);
			DeleteAudioDecoder(audioDecLib,auDec);
			auDec=NULL;
		}
		fclose(inFp);
	} else {
		if(currentIndex<lastIndex) {
			SendMsg(-1,UIMSG_TEXT_ERROR_MSG,(u_int32)"Cannot open file");
		}
		result=ceFormatNotFound;
	}
	if(result!=ceCancelled) {	
		if(result!=ceFormatNotFound && result!=ceFormatNotSupported) {
			jumpDirection=1;
		}
		signal|=S_IDLECHECK;
	}
	ReleaseMutex(&mutex);
}

#include <codMDec.h>
#include "layers_tiny.h"

void CodecServicesCallback(AUDIO_DECODER* auDec,u_int16 samples) {
	if(auDec->cs.playTimeSeconds!=lastSecond) {
		u_int16 perCent;
		lastSecond=auDec->cs.playTimeSeconds;
		if(AttemptMutex(&mutex)==0) {
			SendMsg(currentIndex,UIMSG_U32_PLAY_TIME_SECONDS,lastSecond);
			if(auDec->cs.fileLeft!=0xFFFFFFFFU) {
				perCent=(u_int16)(100.9-(auDec->cs.fileLeft*100.8/auDec->cs.fileSize));
				if(perCent>100) {
					perCent=100;
				}
				SendMsg(lastIndex,UIMSG_U32_PLAY_FILE_PERCENT,perCent);
			}
			ReleaseMutex(&mutex);
		}
	}
	if(auDec->pause!=lastPause) {
		lastPause=auDec->pause;
		SendMsg(-1,UIMSG_S16_PLAYING,lastPause?UIM_PLAYING_PAUSED:UIM_PLAYING_PLAYING);
	}
	if(signal&S_CLOSE_DECODER) {
		auDec->cs.cancel=1;
	}
}

//
// Minimal playlist (room for one named item)
//

static s_int16 MinimalPlaylistSize(register ABSTRACT_PLAYLIST* _) { return 1; }

static u_int16 MinimalPlaylistName(register SINGLE_FILE_PLAYLIST* p,register s_int16 index,register char* buffer,register s_int16 sz) {
	strcpy(buffer,(index==0)?p->name:"");
	return 1;
}

static FILE* MinimalPlaylistOpen(register SINGLE_FILE_PLAYLIST* p,register s_int16 index) {
	return (index==0)?fopen(p->name,"rb"):0;
}

static void MinimalPlaylistDelete(register ABSTRACT_PLAYLIST* _) {}

static const SINGLE_FILE_PLAYLIST singleFilePlaylist={
	{(PLSIZEMETHOD)MinimalPlaylistSize,
	 (PLNAMEMETHOD)MinimalPlaylistName,
	 (PLOPENMETHOD)MinimalPlaylistOpen,
	 (PLRELEASEMETHOD)MinimalPlaylistDelete,
	 PF_AUTOADVANCE|PF_LOOPING}
};

//
// ID3 parser routines
//

s_int32 GetID3Size7(FILE* fp) {
	int i;
	u_int32 res=0;
	for(i=0;i<4;i++) {
		res<<=7;
		res|=fgetc(fp)&0x7F;
	}
	return res;
}

s_int32 GetID3Size8(FILE* fp) {
	int i;
	u_int32 res=0;
	for(i=0;i<4;i++) {
		res<<=8;
		res|=fgetc(fp);
	}
	return res;
}

#define ID3F_UNSYNCHRONIZED	0x80
#define ID3F_EXTENDED		0x40
#define ID3F_EXPERIMENTAL	0x20

typedef struct ID3Tags {
	u_int16 uiMessage;
	char tagStr[4];
} ID3Tags;

#pragma msg 275 off

const ID3Tags id3Tags[]={
	{UIMSG_TEXT_SONG_NAME,"TIT2"},
	{UIMSG_TEXT_ALBUM_NAME,"TALB"},
	{UIMSG_TEXT_ARTIST_NAME,"TPE1"},
	{UIMSG_TEXT_YEAR,"TYER"},
	{UIMSG_VSOS_END_OF_LIST,""}
};

typedef void (*StringDecoder)(register u_int16*,register char);
#define GetStringDecoder(lib) (*(((StringDecoder*)(lib))+2+ENTRY_1))

static void TrimAndSend(register char* string,register UICallback callback,register u_int16 uiMsg) {
	// remove trailing blanks from ID3V1 tag
	char* c=string+29;
	int i=0;
	while(c[0]==' ' && i<30) {
		--c;
		++i;
	}
	c[1]=0;
	callback(-1,uiMsg,(u_int32)string);
}

//
//  DecodeID3: parses ID3v1 and ID3v2 tag information from an audio file.
//  Parsing textual fields (optionally) calls the dynamic library 'Mojibake'
//  which converts different character encodings to UCS-2. If this library
//  is not present, only 8-bit coding will work properly (ASCII/latin-1).
//

void DecodeID3(register FILE* fp,register UICallback callback) {
	if(	fgetc(fp)=='I' && fgetc(fp)=='D' && fgetc(fp)=='3' && 
		((fgetc(fp)+1)>>1)==2 /* (accept v2.3 and 2.4) */ && fgetc(fp)==0x00) {
		/* found ID3v2 tag */
		#ifdef USE_MOJIBAKE
		void* stringDecoderLib=LoadLibrary("MOJIBAKE");
		#endif
		u_int16 flags=fgetc(fp);
		s_int32 id3Left=GetID3Size7(fp);
		if(flags&ID3F_EXTENDED) {
			u_int16 extSize=(u_int16)GetID3Size8(fp);
			fseek(fp,extSize,SEEK_CUR);
			id3Left-=extSize;
		}
		while(id3Left>0) {
			char frameId[4];
			s_int32 frameSize;
			u_int16 frameFlags;
			u_int16 thereIsFrameId=0;
			s_int16 i;
			for(i=0;i<4 && id3Left>0;i++) {
				thereIsFrameId|=(frameId[i]=fgetc(fp));
				id3Left--;
			}
			if(thereIsFrameId && id3Left>=6) {
				ID3Tags* cTag;
				frameSize=GetID3Size8(fp);
				frameFlags=(u_int16)fgetc(fp)<<8;
				frameFlags|=fgetc(fp);
				id3Left-=6;
				if(id3Left<frameSize) {
					frameSize=id3Left;
				}
				id3Left-=frameSize;
				i=0;
				cTag=id3Tags;
				while(cTag->uiMessage!=UIMSG_VSOS_END_OF_LIST) {
					if(!memcmp(cTag->tagStr,frameId,4)) {
						u_int16 s;
						u_int16* write=currTmpStr;
						char coding=fgetc(fp);
						--frameSize;
						s=(u_int16)frameSize;
						if(s>MAX_TMP_STR_LEN-2) {
							s=MAX_TMP_STR_LEN-2;
						}
						while(s-->0) {
							write[0]=fgetc(fp);
							++write;
							--frameSize;
						}
						write[0]=0;
						#ifdef USE_MOJIBAKE
						write[1]=0; // (double termination)
						if(stringDecoderLib) {
							StringDecoder Decode=GetStringDecoder(stringDecoderLib);
							Decode(currTmpStr,coding);
						}
						#endif
						callback(-1,cTag->uiMessage,(u_int32)currTmpStr);
					}
					++cTag;
				}
				fseek(fp,frameSize,SEEK_CUR);
			} else {
				fseek(fp,id3Left,SEEK_CUR);
				id3Left=0;
			}
		}
		#ifdef USE_MOJIBAKE
		if(stringDecoderLib) {
			DropLibrary(stringDecoderLib);
		}
		#endif
	} else { /* Look for ID3v1 tag */
		s_int16 i;
		char* c=(char*)currTmpStr;
		fseek(fp,-128,SEEK_END);
		for(i=0;i<128;i++) {
			*c++=fgetc(fp);
		}
		c=(char*)currTmpStr;
		if(!strncmp(c,"TAG",3)) { /* ID3v1 tag found */
			TrimAndSend(c+3,callback,UIMSG_TEXT_SONG_NAME);
			TrimAndSend(c+33,callback,UIMSG_TEXT_ARTIST_NAME);
			TrimAndSend(c+63,callback,UIMSG_TEXT_ALBUM_NAME);
			c+=93;
			c[4]=0;
			callback(-1,UIMSG_TEXT_YEAR,(u_int32)c);
		}
		fseek(fp,0,SEEK_SET);
	}
}

void QueryPlaylist(register s_int16 index,register UICallback callback) {
	if(playlist->Name(playlist,index,(wchar*)currTmpStr,sizeof(currTmpStr))) {
		FILE* f;
		callback(index,UIMSG_TEXT_LONG_FILE_NAME,(u_int32)currTmpStr);
		f=playlist->Open(playlist,index);
		if(f) {
			DecodeID3(f,callback);
			fclose(f);
		}
	}
}