/*
  Copyright 2008-2014 VLSI Solution Oy. Absolutely no warranty.

  This version supports queueing new files during playback using UART control.
*/  
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <vs1000.h>
#include <audio.h>
#include <mappertiny.h>
#include <minifat.h>
#include <codec.h>
#include <vsNand.h>
#include <player.h>
#include <usblowlib.h>

//#define USE_ENCRYPTION   /* simple encryption -- 77 words */

//#define GAPLESS //try to avoid all delays between files
//#define LOOP_MODE
#define USE_WAV /* Play linear wav files using CodMicroWav from libdev1000. */
#define USE_WAVGOTO

//#define QUEUE_SIZE 100 /*Allow queue -- files can be added in file mode, not while playing. */


//#define EXTCLOCK 13000 /* set to correct clock if not 12MHz. usbmass.c sets clock at first boot */
#define START_IN_FILE_MODE /* do not start to play automatically */
//#define USE_USB /* You get more free code space if you disable USB. */
#define DISABLE_USB //Disable USB code from sdplayer to give room for queue

//#define CHIP_TOTAL_BLOCKS  32768 /*  32768 * 512 bytes = 16MB */
//#define CHIP_TOTAL_BLOCKS  8192 /*  8192 * 512 bytes = 4M (Winbond 25X32) */
#define CHIP_TOTAL_BLOCKS  4096 /*  4096 * 512 bytes = 2M (Winbond 25X16) */

/* Set aside some blocks for VS1000 boot code (and optional parameter data) .
   Note that this is different than in vs1000 button cell player and some
   other example codes. */
#define RESERVED_BLOCKS 64
#define LOGICAL_DISK_BLOCKS  (CHIP_TOTAL_BLOCKS-RESERVED_BLOCKS)

#define GPIO0_PULLUPS   0x1b80 // Audio module has pull-ups in these pins

#define PLAYER_WAIT_FOR_COMMAND 0
#define PLAYER_CONTINUOUS_PLAY  1
s_int16 playerMode; // = PLAYER_CONTINUOUS_PLAY;
void CheckSd(void); //called by uart.c and gpio.c to check SD insert/remove

#include <vstypes.h>
/* generic support functions */
void putstrp(register __i0 u_int16 *packedStr);
unsigned short atou(register __i2 const char *s);
unsigned long atol(register __i2 const char *s);

#define RX_BUF_SIZE 64
/* Note: the data in uartRxBuffer[] is bytes, not words. */
extern __y u_int16 uartRxBuffer[RX_BUF_SIZE];
extern volatile __y u_int16 * __y uartRxWrPtr;
extern __y u_int16 * __y uartRxRdPtr;

int UartFill(void);
int UartGetByte(void);
void MyRxIntCommand(void);

/** Player support routines, see uart.c. */
void UartLoadCheck(struct CodecServices *cs, s_int16 n);
void UartIdleHook(void);
void UartInit(void); // Redirect RX interrupt etc.

extern struct CodecServices cs;

#ifdef START_IN_FILE_MODE
s_int16 playerMode = PLAYER_WAIT_FOR_COMMAND;
#else
s_int16 playerMode = PLAYER_CONTINUOUS_PLAY;
#endif

#ifdef QUEUE_SIZE
s_int16 queued = 0;
u_int16 queue[QUEUE_SIZE];
#endif


#ifdef DISABLE_USB
#undef USE_USB
#endif

//#define USE_MULTIPLE_BLOCK_WRITE /*doubles write speeds on SD cards*/
#define USE_MULTIPLE_BLOCK_READ
//#define USE_DEBUG
//#define DEBUG_LEVEL 3

#define USE_HC      /* Support SD-HC cards. */
//#define PATCH_EJECT    /* 'eject' shuts down power */
#define PATCH_UNIT_READY /* detects card removed during USB mode */
#define PATCH_SENSE      /* patches REQUEST_SENSE */


#include <dev1000.h>

/*
  TODO: use multiple block read and multiple block write
  TODO: readahead using multiple block read?
  TODO: multiple block write would need error handling, possibly rewrites

  Start Block Tokens and Stop Tran Token:
  Single block read, single block write, multiple block read:
  - start block 0xfe
  Multiple block write (CMD25):
  - start block 0xfc
  - stop transmission 0xfd

  Multiple block read (CMD18):
  - stop transmission: STOP_TRAN = CMD12

  state transitions:
  ok -> read>ok
  ok -> (0xfe)write>ok
  ok -> (0xfe)write>writewait -> ok

  state transitions:
  ok -> readm(0xfe) -> readm(0xfe) -(STOP_TRAN)-> ok
  ok -> (0xfc)writem ->writemw -> (0xfc)writem -> writemw -(0xfd)-> ok

 */


enum mmcState {
    mmcNA = 0,
    mmcOk = 1,
    mmcWriteWait = 2, /* waiting for write to finish */
};
struct {
    enum mmcState state;
    s_int16 errors;
    s_int16 hcShift;
    u_int32 blocks;
#ifdef USE_MULTIPLE_BLOCK_WRITE
    u_int32 nextWriteBlock;
#endif
#ifdef USE_MULTIPLE_BLOCK_READ
  u_int32 nextReadBlock;
#endif
} mmc;


#ifdef USE_DEBUG
static char hex[] = "0123456789ABCDEF";
void puthex(u_int16 d) {
    register int i;
    char mem[5];
    for (i=0;i<4;i++) {
	mem[i] = hex[(d>>12)&15];
	d <<= 4;
    }
    mem[4] = '\0';
    fputs(mem, stdout);
}
#endif


extern struct FsNandPhys fsNandPhys;
extern struct FsPhysical *ph;
extern struct FsMapper *map;
extern struct Codec *cod;
extern struct CodecServices cs;
//extern u_int16 codecVorbis[];


s_int16 InitializeMmc(s_int16 tries) {
    register s_int16 i, cmd;
    mmc.state = mmcNA;
    mmc.blocks = 0;
    mmc.errors = 0;
#ifdef USE_MULTIPLE_BLOCK_WRITE
    mmc.nextWriteBlock = 0xffffffff;
#endif
#ifdef USE_MULTIPLE_BLOCK_READ
    mmc.nextReadBlock = 0xffffffff;
#endif

#if DEBUG_LEVEL > 1
    puthex(clockX);
    puts(" clockX");
#endif
 tryagain:
    IdleHook();

    mmc.hcShift = 9;
    if (tries-- <= 0) {
	return ++mmc.errors;
    }
    for (i=0; i<512; i++) {
	SpiSendClocks();
    }

    /* MMC Init, command 0x40 should return 0x01 if all is ok. */
    i = MmcCommand(MMC_GO_IDLE_STATE/*CMD0*/|0x40,0);
    if (i != 1) {
#if DEBUG_LEVEL > 1
	puthex(i);
	puts(" Init");
#endif
	BusyWait10();
	goto tryagain;//continue; /* No valid idle response */
    }
    cmd = MMC_SEND_OP_COND|0x40;
#ifdef USE_HC
    /*CMD8 is mandatory before ACMD41 for hosts compliant to phys. spec. 2.00*/
    i = MmcCommand(MMC_SEND_IF_COND/*CMD8*/|0x40,
		   0x00000122/*2.7-3.6V*/); /*note: 0x22 for the right CRC!*/
#if DEBUG_LEVEL > 2
    puthex(i);
    puts("=IF_COND");
#endif
    if (i == 1) {
	/* MMC answers: 0x05 illegal command, v2.0 SD(HC-SD) answers: 0x01 */
	/* Should we read the whole R7 response? */
#if DEBUG_LEVEL > 1
	//SpiSendReceiveMmc(-1, 32);
	puthex(SpiSendReceiveMmc(-1, 16));
	puthex(SpiSendReceiveMmc(-1, 16));
	puts("=R7");
#else
	SpiSendReceiveMmc(-1, 32); /*read the whole response*/
#endif
	cmd = 0x40|41; /* ACMD41 - SD_SEND_OP_COND */
    }
#endif /*USE_HC*/


#if 0
    do {
	i = MmcCommand(MMC_READ_OCR/*CMD58*/|0x40, 0);
#if DEBUG_LEVEL > 1
	puthex(i);
	puthex(SpiSendReceiveMmc(-1, 16));
	puthex(SpiSendReceiveMmc(-1, 16));
	puts("=READ_OCR");
#endif
    } while (0);
#else
    /* Some cards require CMD58 after CMD8 -- does this still work with MMC?*/
    MmcCommand(MMC_READ_OCR/*CMD58*/|0x40, 0);
    SpiSendReceiveMmc(-1, 16);
    SpiSendReceiveMmc(-1, 16);
#endif

    /* MMC Wake-up call, set to Not Idle (mmc returns 0x00)*/
    i = 0; /* i is 1 when entered .. but does not make the code shorter*/
    while (1) {
	register int c;
#ifdef USE_HC
	if (cmd == (0x40|41)) {
	    c = MmcCommand(0x40|55/*CMD55*/,0);
#if DEBUG_LEVEL > 2
	    puthex(c);
	    puts("=CMD55");
#endif
	}
#endif

	c = MmcCommand(cmd/*MMC_SEND_OP_COND|0x40*/,
#ifdef USE_HC
		       /* Support HC for SD, AND for MMC! */
		       //i ? 0x40100000UL : 0x00000000UL
		       0x40100000UL
#else
		       0
#endif
	    );

#if DEBUG_LEVEL > 1
	puthex(c);
	puts("=ACMD41");
#endif

	if (c == 0) {
#if DEBUG_LEVEL > 1
	    puts("got 0");
#endif
	    break;
	}
	//BusyWait10();
	if (++i >= 25000 /*Large value (12000) required for MicroSD*/
	    || c != 1) {
#if DEBUG_LEVEL > 1
	    puthex(c);
	    puts(" Timeout 2");
#endif
	    goto tryagain; /* Not able to power up mmc */
	}
    }
#ifdef USE_HC
    i = MmcCommand(MMC_READ_OCR/*CMD58*/|0x40, 0);
#if DEBUG_LEVEL > 1
    puthex(i);
    puts("=READ_OCR");
#endif
    if (//cmd == (0x40|41) &&
	i == 0) {
	if (SpiSendReceiveMmc(-1, 16) & (1<<(30-16))) {
	    /* OCR[30]:CCS - card capacity select */
	    /* HighCapacity! */
#if DEBUG_LEVEL > 1
	    puts("=HC");
#endif
	    mmc.hcShift = 0;
	}
	/*read all of the OCR data to make some cards happy */
	SpiSendReceiveMmc(-1, 16);
    }
#endif /*USE_HC*/

#ifdef USE_USB
    if (MmcCommand(MMC_SEND_CSD/*CMD9*/|0x40, 0) == 0) {
	register s_int16 *p = (s_int16 *)minifatBuffer;
	register int t = 640;
	while (SpiSendReceiveMmc(0xff00, 8) == 0xff) {
	    if (t-- == 0) {
#if DEBUG_LEVEL > 1
		puts("Timeout 3");
#endif
		goto tryagain;
	    }
	}
	for (i=0; i<8; i++) {
	    *p++ = SpiSendReceiveMmc(-1, 16);
#if DEBUG_LEVEL > 1
	    puthex(p[-1]);
#endif
/*
            00 00 11 11 222 2 3 3334444555566667777
  64M  MMC: 8C 0E 01 2A 03F 9 8 3D3F6D981E10A4040DF
            CSD  NSAC   CCC  flg       MUL+E
	      TAAC  SPEE   RBL  C_SIZE(>>2=FDB)
	                        33 333333333344 444 444 444 444 445
	                        00 111101001111 110 110 110 110 011 000000111
				   C_SIZE                       MULT
  256M MMC+ 90 5E 00 2A 1F5 9 8 3D3EDB683FF9640001F=CSD (Kingston)
  128M SD:  00 26 00 32 175 9 8 1E9F6DACF809640008B=CSD (DANE-ELEC)

  4G SD-HC: 40 7F 0F 32 5B5 9 8 0001E44 7F801640004B
            00 00 11 11 222 2 3 3334444 555566667777


 */
	}
	if ((minifatBuffer[0] & 0xf000) == 0x4000) {
	    /* v2.0 in 512kB resolution */
	  //puts("v2");
	    mmc.blocks = 
		(((((u_int32)minifatBuffer[3]<<16) | minifatBuffer[4])
		 & 0x3fffff) + 1) << 10;
	} else {
	    /* v1.0 */
	    register s_int32 c_size = (((minifatBuffer[3] & 0x03ff)<<2) |
				       ((minifatBuffer[4]>>14) & 3)) + 1;
	    register s_int16 c_mult = (((minifatBuffer[4] & 3)<<1) |
				       ((minifatBuffer[5]>>15) & 1));
	    //puts("v1");
	    c_size <<= (2-9) + c_mult + (minifatBuffer[2] & 15);
	    mmc.blocks = c_size;
	}
#if DEBUG_LEVEL > 1
	puts("=CSD");
	puthex(mmc.blocks>>16);
	puthex(mmc.blocks);
	puts("=mmcBlocks");
#endif
    }
#endif/*USE_USB*/


#if DEBUG_LEVEL > 4
    if (MmcCommand(MMC_SEND_CID/*CMD10*/|0x40, 0) == 0) {
	register s_int16 *p = minifatBuffer;
	register int t = 3200;
	while (SpiSendReceiveMmc(0xff00, 8) == 0xff) {
	    if (t-- == 0)
		goto tryagain;
	}
	for (i=0; i<8; i++) {
	    *p++ = SpiSendReceiveMmc(-1, 16);
#if DEBUG_LEVEL > 1
	    puthex(p[-1]);
#endif
/*
       Man     Productname   serial#   date
          App             rev        res    crc7+stop
  4G:  1D 4144 0000000000 00 0000157A 0 06A E3
  64M: 02 0000 53444D3036 34 40185439 0 C77 75
       00 0011 1122223333 44 44555566 6 677 77
*/
	}
#if DEBUG_LEVEL > 1
	puts("=CID");
#endif
    }
#endif

#if 0
    /* Set Block Size of 512 bytes -- default for at least HC */
    /* Needed by MaxNova S043618ATA 2J310700 MV016Q-MMC */
    /* Must check return value! (MicroSD restart!) */
    {
	register int c;
	if ((c = MmcCommand(MMC_SET_BLOCKLEN|0x40, 512)) != 0) {
#if DEBUG_LEVEL > 1
	    puthex(c);
	    puts(" SetBlockLen failed");
#endif
	    goto tryagain;
	}
    }
#endif
    /* All OK return */
    //mmc.errors = 0;
    mmc.state = mmcOk;
#ifdef USE_USB
    map->blocks = mmc.blocks;
#endif
    return 0;//mmc.errors;
}

struct FsMapper *FsMapMmcCreate(struct FsPhysical *physical,
				u_int16 cacheSize);
u_int16 FsMapMmcRead(struct FsMapper *map, u_int32 firstBlock,
		     u_int16 blocks, u_int16 *data);
u_int16 FsMapMmcWrite(struct FsMapper *map, u_int32 firstBlock,
		      u_int16 blocks, u_int16 *data);

auto u_int16 MyReadDiskSector(register __i0 u_int16 *buffer,
			      register __reg_a u_int32 sector) {
    register s_int16 i;
    register u_int16 t = 65535;

    if (mmc.state == mmcNA || mmc.errors) {
	cs.cancel = 1;
	return 5;
    }
#ifdef USE_MULTIPLE_BLOCK_WRITE
    /* with "if" it is a little faster */
    //if (mmc.nextWriteBlock != 0xffffffff)
    {
	/* Finalize write mode */
	FsMapMmcWrite(NULL, 0, 0, NULL);
    }
#endif
#ifdef USE_MULTIPLE_BLOCK_READ
    if (sector != mmc.nextReadBlock) {
      if (mmc.nextReadBlock != 0xffffffff) {
	i = MmcCommand(MMC_STOP_TRANSMISSION|0x40, 0);
	/* 00 7f 00 ff ff ff ff .. */
	//puthex(i);
	t = 65535;
	do {
	  i = SpiSendReceiveMmc(0xff00, 8);
	  /*TODO: timeout?*/
	} while (i != 0xff && --t != 0);
	SpiSendClocks();
      }
#ifdef USE_HC
      i = MmcCommand(MMC_READ_MULTIPLE_BLOCK|0x40,(sector<<mmc.hcShift));
#else
      i = MmcCommand(MMC_READ_MULTIPLE_BLOCK|0x40,(sector<<9));
#endif
      mmc.nextReadBlock = sector+1;
    }
    /* wait for data token */
    do {
      i = SpiSendReceiveMmc(0xff00,8);
    } while (i == 0xff && --t != 0);
    if (i != 0xfe) {
	memset(buffer, 0, 256);
	if (i > 15 /*unknown error code*/) {
	    mmc.errors++;
	} else {
	    /* data retrieval failed:
	       D4-D7 = 0
	       D3 = out of range
	       D2 = Card ECC Failed
	       D1 = CC Error
	       D0 = Error
	     */
	}
	SpiSendClocks();
	return 1;
    }
    for (i=0; i<512/2; i++) {
	*buffer++ = SpiSendReceiveMmc(0xffff,16);
    }
    SpiSendReceiveMmc(0xffff,16); /* discard crc */
#else /*USE_MULTIPLE_BLOCK_READ*/

#if 0 && defined(USE_DEBUG)
    puthex(sector>>16);
    puthex(sector);
    puts("=ReadDiskSector");
#endif
#ifdef USE_HC
    MmcCommand(MMC_READ_SINGLE_BLOCK|0x40,(sector<<mmc.hcShift));
#else
    MmcCommand(MMC_READ_SINGLE_BLOCK|0x40,(sector<<9));
#endif
    do {
	i = SpiSendReceiveMmc(0xff00,8);
    } while (i == 0xff && --t != 0);

    if (i != 0xfe) {
	memset(buffer, 0, 256);
	if (i > 15 /*unknown error code*/) {
	    mmc.errors++;
	} else {
	    /* data retrieval failed:
	       D4-D7 = 0
	       D3 = out of range
	       D2 = Card ECC Failed
	       D1 = CC Error
	       D0 = Error
	     */
	}
	SpiSendClocks();
	return 1;
    }
    for (i=0; i<512/2; i++) {
	*buffer++ = SpiSendReceiveMmc(0xffff,16);
#if 0
	puthex(buffer[-1]);
	if ((i & 15) == 15)
	    puts("");
#endif
    }
    SpiSendReceiveMmc(0xffff,16); /* discard crc */

    /* generate some extra SPI clock edges to finish up the command */
    SpiSendClocks();
    SpiSendClocks();
#endif /*else USE_MULTIPLE_BLOCK_READ*/

    /* We force a call of user interface after each block even if we
       have no idle CPU. This prevents problems with key response in
       fast play mode. */
    IdleHook();
    return 0; /* All OK return */
}

#ifdef USE_MULTIPLE_BLOCK_WRITE
s_int16 FsMapMmcFlush(struct FsMapper *map, u_int16 hard) {
    /* Finalize write mode */
    FsMapMmcWrite(NULL, 0, 0, NULL);
    return 0;
}
#endif

#ifdef USE_USB
const struct FsMapper mmcMapper = {
    0x010c, /*version*/
    256,    /*blocksize in words*/
    0,      /*blocks -- read from CSD*/
    0,      /*cacheBlocks*/
    NULL/*FsMapMmcCreate*/,
    FsMapFlNullOk,//RamMapperDelete,
    FsMapMmcRead,
    FsMapMmcWrite,
    NULL,//FsMapFlNullOk,//RamMapperFree,
#ifdef USE_MULTIPLE_BLOCK_WRITE
    FsMapMmcFlush,
#else
    FsMapFlNullOk,//RamMapperFlush,
#endif
    NULL /* no physical */
};
#endif/*USE_USB*/

#if 0
struct FsMapper *FsMapMmcCreate(struct FsPhysical *physical,
				u_int16 cacheSize) {
    PERIP(GPIO0_MODE) &= ~(MMC_MISO|MMC_CLK|MMC_MOSI|MMC_XCS);
    PERIP(GPIO0_DDR) = (PERIP(GPIO0_DDR) & ~(MMC_MISO))
	| (MMC_CLK|MMC_MOSI|MMC_XCS);
    PERIP(GPIO0_ODATA) = (PERIP(GPIO0_ODATA) & ~(MMC_CLK|MMC_MOSI))
	| (MMC_XCS | GPIO0_CS1); /* NFCE high */
#if DEBUG_LEVEL > 1
    puts("Configured MMC pins\n");
#endif
    memset(&mmc, 0, sizeof(mmc));
    return &mmcMapper;
}
#endif

#ifdef USE_USB
u_int16 FsMapMmcRead(struct FsMapper *map, u_int32 firstBlock,
		     u_int16 blocks, u_int16 *data) {
    /* We know this is called one block at a time by the USB ROM firmware,
       so only prepare for that and save 20 words. */
    /*remove sign extension: 4G -> 8BG limit*/
    if (MyReadDiskSector(data, firstBlock & 0x00ffffff))
	return 0; /* probably MMC detached */
    return 1;
}
u_int16 FsMapMmcWrite(struct FsMapper *map, u_int32 firstBlock,
		      u_int16 blocks, u_int16 *data) {
    /* We know this is called one block at a time by the USB ROM firmware,
       so only prepare for that and save 14 words. */
    firstBlock &= 0x00ffffff; /*remove sign extension: 4G -> 8BG limit*/
#ifdef USE_MULTIPLE_BLOCK_WRITE
    if (!data || firstBlock != mmc.nextWriteBlock) {
	/* Finalize write mode */
	if (mmc.nextWriteBlock != 0xffffffff) {
	    mmc.nextWriteBlock = 0xffffffff;
	    SpiSendReceiveMmc(0xfd00, 8); /* End of write */
	    do {
		/*TODO: timeout?*/
	    } while ( SpiSendReceiveMmc(0xff00U, 8) != 0x00ffU);
	    SpiSendClocks();
	}
	if (!data)
	    return 0;
    }
#endif

    {
	register u_int16 c;
	if (mmc.state == mmcNA || mmc.errors)
	    return 0;
#ifdef USE_MULTIPLE_BLOCK_WRITE
	if (mmc.nextWriteBlock == 0xffffffff) {
	    mmc.nextWriteBlock = firstBlock;
#ifdef USE_HC
	    c = MmcCommand(MMC_WRITE_MULTIPLE_BLOCK|0x40, firstBlock<<mmc.hcShift);
#else
	    c = MmcCommand(MMC_WRITE_MULTIPLE_BLOCK|0x40, firstBlock<<9);
#endif
	    //wait for BUSY token, if you get 0x01(idle), it's an ERROR!
	    while (c != 0x00) {
		if (c == 0x01) {
		    mmc.errors++;
		    goto out;
		}
		c = SpiSendReceiveMmc(0xff00, 8);
		/*TODO: timeout?*/
	    }
	}
	mmc.nextWriteBlock++;
	SpiSendReceiveMmc(0xfc00, 8);
#else/*MULTIPLE*/

#ifdef USE_HC
	c = MmcCommand(MMC_WRITE_BLOCK|0x40, firstBlock<<mmc.hcShift);
#else
	c = MmcCommand(MMC_WRITE_BLOCK|0x40, firstBlock<<9);
#endif
	//wait for BUSY token, if you get 0x01(idle), it's an ERROR!
	while (c != 0x00) {
	    if (c == 0x01) {
		mmc.errors++;
		goto out;
	    }
	    c = SpiSendReceiveMmc(0xff00, 8);
	    /*TODO: timeout?*/
	}
	SpiSendReceiveMmc(0xfffe, 16);
#endif/*else MULTIPLE*/

	for (c=0; c<256; c++) {
	    SpiSendReceiveMmc(*data++, 16);
	}
	SpiSendReceiveMmc(0xffff, 16); /* send dummy CRC */

	if (map) {
	    /* wait until MMC write finished */
	    do {
		/*TODO: timeout?*/
	    } while ( SpiSendReceiveMmc(0xff00U, 8) != 0x00ffU);
	} else {
	    mmc.state = mmcWriteWait;
	    /* MUST not use SpiSendClocks():
	       some MMC/SD's don't support interrupted XCS. */
	    return 0; /*Can write only one sector this way.*/
	}
    }
 out:
#ifdef USE_MULTIPLE_BLOCK_WRITE
    if (mmc.nextWriteBlock != 0xffffffff) {
    } else {
	SpiSendClocks();
    }
#else
    SpiSendClocks();
#endif
    return 1;
}
#endif/*USE_USB*/

#ifdef USE_USB
#if defined(PATCH_EJECT) || defined(PATCH_UNIT_READY) || defined(PATCH_SENSE)
#include <scsi.h>
#endif
#ifdef PATCH_EJECT
int startUnit = 1;
#endif

/*
  We replace the MassStorage() functionality.
  Uses ScsiTestUnitReady() to check the MMC/SD removal.
  Only appears as a mass storage device when MMC/SD is present.
 */
auto void MyMassStorage(void) {
#ifdef USE_DEBUG
  puts("MyMassStorage");
#endif
  if (mmc.state == mmcNA || mmc.errors) {
#ifdef USE_DEBUG
      puts("No MMC -> no USB MassStorage");
#endif
      return;
  }

  PowerSetVoltages(&voltages[voltCoreUSB]);
  /** a 2.5ms-10ms delay to let the voltages change (rise) */
//  BusyWait10();

  LoadCheck(NULL, 1); /* 4.0x clock required! */
  InitUSB(USB_MASS_STORAGE);  

#ifdef USE_DEBUG
  puts("USB loop");
#endif
  while (mmc.state != mmcNA && mmc.errors == 0) {
    USBHandler();

    /* If no SOF in such a long time.. */
    if (USBWantsSuspend()) {
      /* Do we seem to be detached?
	 Note: Now also requires USBWantsSuspend() to return TRUE! */
      if (USBIsDetached()) {
#ifdef USE_DEBUG
	puts("Detached");
#endif
	break;
      }
#ifdef USE_DEBUG
      puts("Suspend");
#endif
      /*Suspend is not supported anyway*/
      USB.lastSofTime = ReadTimeCount();
    }
#ifdef PATCH_EJECT
    if (startUnit == 2) {
	break;
    }
#endif
  }
#ifdef USE_DEBUG
  puts("Leaving USB");
#endif
//  hwSampleRate = 1; /* Forces FREQCTL update at next SetRate() */
  PERIP(SCI_STATUS) &= ~SCISTF_USB_PULLUP_ENA; /* Take us 'out of the bus' */
  PERIP(USB_CONFIG) = 0x8000U; /* Reset USB */
#ifdef USE_MULTIPLE_BLOCK_WRITE
  map->Flush(map, 1);
#endif

  /* Set player voltages. No need to wait for the voltages to drop. */
  PowerSetVoltages(&voltages[voltCorePlayer]);
#ifdef PATCH_EJECT
  /* If 'eject', power down */
  if (startUnit == 2) {
#ifdef UART_CONTROL
      putstrp("\pEjected - Power Off\n\n");
#endif/*UART_CONTROL*/
      PowerOff();
  }
#endif
}
#endif /*USE_USB*/


#if defined(USE_WAV) && defined(USE_WAVGOTO)
 #include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <codec.h>
#include <audio.h>
#include <minifat.h>
#include "vs1000.h"

#include <usblowlib.h> //for SwapWord
#include <dev1000.h>

#define MAKE_ID(c,d,a,b) (((u_int32)(a)<<24)|((u_int32)(b)<<16)|((c)<<8)|(d))
#define	WAVE_FORMAT_PCM		(0x0001)
#define	WAVE_FORMAT_MULAW	(0x0007)
#define	WAVE_FORMAT_IMA_ADPCM		(0x0011)
/** Riff 'f','m','t',' ' TAG format. This is a little-endian format. */
struct WavRiffFormatTag {
  u_int16 formatTag;		/**< WAV subformat */
  u_int16 channels;		/**< Number of audio channels */
  u_int32 samplesPerSecond;	/**< Sample rate -- mixed-endian! */
  u_int32 avgBytesPerSec;	/**< Average data rate */
  u_int16 blockAlign;		/**< Block align if format is block oriented */
  u_int16 bitsPerSample;	/**< Number of bits per audio sample */
};

enum CodecError CodGotoWavDecode(struct Codec *cod,
				 struct CodecServices *cs,
				 const char **errorString);

extern u_int16 ogg[398];
extern u_int16 codecVorbis[];

volatile s_int32 longGoTo = -1; /* ms */

/*
  g_yprev0 is not suitable because it is in address zero
 */
enum CodecError CodGotoWavDecode(struct Codec *dummy_cod,
				 struct CodecServices *dummy_cs,
				 const char **errorString) {
    u_int32 dataStart;
    u_int32 riffLeft, dat;
    __y s_int16 w;
    __y s_int16 cnt = 0;
    __y u_int16 fmtTag = 0;
#ifdef IMA_ENABLE
    __y u_int16 imaBA = 0;
#endif
    register u_int32 bytesLeft;
    if (!cs.Seek)
	return ceOtherError; /* Seek is required! */

#if 1
    codecVorbis[16+1] = 0; /* clr the block size so earSpeaker can be on.. */
#endif
    cs.Read(&cs, (void *)&dat, 0, 4);
    if (dat != MAKE_ID('R','I','F','F')) {
	return ceFormatNotFound;
    }
    cs.playTimeSamples = cs.playTimeSeconds = 0;
    cs.gain = -12;

    /* RIFF */
    cs.Read(&cs, (void *)&dat, 0, 4);
    riffLeft = WavSwap32(dat) -4;
    cs.Read(&cs, (void *)&dat, 0, 4);
    if (dat != MAKE_ID('W','A','V','E')) {
      //puts("ceFormatNotSupported");
	return ceFormatNotSupported;
    }
    while (riffLeft >= 8 && !cs.cancel) {
	u_int32 name;
	__y u_int16 inBuf;

	cs.Read(&cs, (void *)&name, 0, 4);
	cs.Read(&cs, (void *)&dat, 0, 4);
	bytesLeft = WavSwap32(dat);
	if (name == MAKE_ID('d','a','t','a')) {
	    break;
	}
	inBuf = (bytesLeft > 16) ? 16 : bytesLeft;
	riffLeft -= 8 + bytesLeft;
	if (inBuf) {
	    cs.Read(&cs, g_others, 0, inBuf);
	    bytesLeft -= inBuf;
	}
	if (name == MAKE_ID('f','m','t',' ')) {
	    /* Is there enough minimum data? */
	    if (inBuf >= sizeof(struct WavRiffFormatTag)*(2/sizeof(u_int16))) {
		register struct WavRiffFormatTag *fmt =
		    (struct WavRiffFormatTag *)g_others;
		fmtTag = SwapWord(fmt->formatTag);
		cs.channels = SwapWord(fmt->channels);
		cs.sampleRate = SwapWord/*Swap32Mix*/(fmt->samplesPerSecond);
		/* cs.Output() handles sample rate */
		if (cs.channels > 2)
		    return ceFormatNotSupported;
#if 0
		puthex(cs.channels);
		puthex(cs.sampleRate);
		puts("chn,rate");
		puthex(SwapWord(fmt->formatTag));
		puthex(SwapWord(fmt->bitsPerSample));
		puts("fmt,bps");
#endif
		w = SwapWord(fmt->bitsPerSample);
		if (fmtTag == WAVE_FORMAT_PCM ||
		    fmtTag == 0xfffe /*WAV with extended header*/) {
//puts("linear");
		} else {
		    return ceFormatNotSupported;
		}
	    }
	}
	cs.Seek(&cs, bytesLeft, SEEK_CUR);
    }
 data:
    dataStart = cs.Tell(&cs); /* for cs.goTo */
    while (bytesLeft) {
	int rd;
	{
	    rd = (sizeof(g_others) & ~3); /*in bytes (half of the buffer)*/
	    if (w == 16)
		rd <<= 1; /* for 16-bit use the whole buffer */
	    if (rd > bytesLeft)
		rd = bytesLeft;

	    rd = cs.Read(&cs, g_others, 0, rd);
	    bytesLeft -= rd;
	    if (!rd)
		break;
	    if (w == 8) {
	      /* 8-bit linear */
	      register s_int16 i = 0;
	      register u_int16 *wp = g_others + rd;
	      register u_int16 *rp = (g_others-1)+rd/2;
	      for (i=0; i<rd; i+=2) {
		register u_int16 t = *rp ^ 0x8080U;
		*--wp = (t << 8);
		*--wp = (t & 0xFF00U);
		rp--;
	      }
	    } else {
		/* 16-bit linear */
		register int i;
		register u_int16 *wp = g_others;
		for (i=0; i<rd; i+=2) {
		    *wp = (*wp << 8) | (*wp >> 8);
		    wp++;
		}
		rd >>= 1;
	    }
	    if (cs.channels == 2)
		rd >>= 1;
	}
	/* update playtime: rd is the number of samples to output */
	if ((cs.playTimeSamples += (s_int32)cs.fastForward * rd) > cs.sampleRate) {
	    cs.playTimeSamples -= cs.sampleRate;
	    cs.playTimeSeconds++;
	}
	cs.Output(&cs, (s_int16 *)g_others, rd);

#if 1   /*implement goTo -- slow down the reaction artificially */
	if (cnt > 0) {
	    cnt -= rd;
#if 0
	} else if (cs.goTo != 0xffffU) {
	  s_int32 pos = dataStart
	    + ((s_int32)cs.goTo * cs.sampleRate * (cs.channels * w / 8));
	  cs.playTimeSamples = 0;
	  cs.playTimeSeconds = cs.goTo;
	  cnt = cs.sampleRate/4; /* next goTo allowed after 1/4 sec */
	  cs.goTo = -1;
	  //if (pos >= cs.fileSize) {
	  //break;
	  //}
	  bytesLeft += cs.Tell(&cs) - pos;
	  cs.Seek(&cs, pos, SEEK_SET);
#else
	} else if (longGoTo != -1L) {
	  s_int32 pos = dataStart
	    + (longGoTo * cs.sampleRate / 1000 * (cs.channels * w / 8));
	  cs.playTimeSamples = 0;
	  cs.playTimeSeconds = cs.goTo;
	  cnt = cs.sampleRate/4; /* next goTo allowed after 1/4 sec */
	  longGoTo = -1L;
	  if (pos >= cs.fileSize) {
	    break;
	  }
	  bytesLeft += cs.Tell(&cs) - pos;
	  cs.Seek(&cs, pos, SEEK_SET);
#endif
	}
#endif
#if 0 /* implement fast play mode -- not very good for high bitrates,
	 should play longer pieces or window the samples */
	else if (cs.fastForward > 1) {
	    s_int32 seek;
	    {
		seek = (cs.fastForward-1) * (s_int32)rd;
	    }
	    if (seek > bytesLeft)
		seek = bytesLeft;
	    cs.Seek(&cs, seek, SEEK_CUR);
	    bytesLeft -= seek;
	}
#endif
	if (cs.cancel) {
	    cs.cancel = 0;
	    return ceOk;//ceCancelled;
	}
    }
    //puts("ceOk");
    return ceOk;
}

#endif /*defined(USE_WAV) && defined(USE_WAVGOTO)*/



#ifdef GAPLESS

#ifndef COD_VORBIS_GENERAL_H
#define COD_VORBIS_GENERAL_H
#include <vstypes.h>
#include <codec.h>
/** Maximum number of channels handled by the VORBIS codec. */
#define COD_VORBIS_MAX_CHANNELS 2
/**
   Vorbis Codec extensions to the basic Codec structure.
   Doesn't yet allow re-entrancy.
*/
struct CodecVorbis {
  /** Basic Codec structure. */
  struct Codec c;
  /** Temporary flag until re-entrancy is corrected. */
  s_int16 used;
  /** Internal variables */
  u_int16 state;
  /* ---------------- setup - -------------------- */
  u_int16 audioChannels;
  u_int32 audioSampleRate;
  u_int32 bitRateMaximum;
  u_int32 bitRateNominal;
  u_int32 bitRateMinimum;
  u_int16 blockSize[2];
  const u_int16 __y *window[2];
  u_int16 codeBooks;
  struct CodeBook *codeBook;
  u_int16 floors;
  struct Floor1 __y *floor;
  u_int16 residues;
  struct Residue *residue;
  u_int16 maps;
  struct Map *map;
  u_int16 modes;
  struct Mode *mode;
  /* ---------------- decode -------------------- */
  /** Mode number */
  u_int16 modeNumber;
  /** Type of previous window */
  s_int16 prevWinType;
  /** Byte offset to the point where actual AUDIO data begins */
  u_int32 audioBegins;
  /** How many bits downshifted for last frame. */
  s_int16 oldFrameDownshift[COD_VORBIS_MAX_CHANNELS];
  /** How many bits could have been downshifted for last frame. */
  s_int16 oldFramePotentialDownshift[COD_VORBIS_MAX_CHANNELS];
};
#endif /* !COD_VORBIS_GENERAL_H */
extern struct CodecVorbis codecVorbis;
struct Codec *CodVorbisCreate(void);
extern u_int16 ogg[];
//extern struct CodVOgg ogg;
extern s_int16 vFirstFrame;
auto s_int16 OggSeekHeader(register u_int32 pos);
#endif/*GAPLESS*/



#ifdef USE_WAV
#include <codecmicrowav.h>
#include <dev1000.h>
extern struct CodecServices cs;
extern struct Codec *cod;
#endif

#ifdef LOOP_MODE
/* For seamless playing of the same file in a loop. These are
   not normally used, so we don't bother including codecvorbis.h . */
auto s_int16 CodVorbisGoTo(register struct CodecVorbis *cv,
			   register u_int16 seconds);
int loopMode = 0, leaveLoop = 0;
#endif/*LOOP_MODE*/


#ifdef USE_ENCRYPTION
#define FIRMWARE_KEY 0x1234 /* Must equal the one in encryptfile.c */
auto u_int16 MapperReadDiskSector(register __i0 u_int16 *buffer,
				  register __reg_a u_int32 sector);
auto u_int16 DecryptReadDiskSector(register __i0 u_int16 *buffer,
				   register __reg_a u_int32 sector) {
  u_int16 res;
  {
    //__y u_int32 f = sector;
    //res = map->Read(map, f, 1, buffer);
    res = MyReadDiskSector(buffer, sector);
  }
  {
      register int i;
      /* Use file position and file size as part of the key. */
      register s_int16 rnd_seed =
	  (s_int16)((s_int32)(minifatInfo.filePos ^ minifatInfo.fileSize)>>9)
	  ^ (s_int16)minifatInfo.fileSize ^ FIRMWARE_KEY;
      for (i=0;i<256;i++) {
	  rnd_seed = rnd_seed * 16217 + 9043;
	  *buffer++ ^= rnd_seed;
      }
  }
  return res;
}

u_int16 DecryptCsRead(struct CodecServices *cs, u_int16 *data,
		      u_int16 firstOdd, u_int16 bytes) {
  static u_int16 tmp[256];
  s_int16 read = 0;

  while (bytes) {
    u_int32 filePos = cs->fileSize - cs->fileLeft;
    u_int16 byt = filePos & 511, len;
    len = bytes;
    if (len > 512-byt) { /* Spans the sector boundary. */
      len = 512-byt;
    }
    read += CsRead(cs, tmp+(byt/2), (byt&1), len);
    {
      register __d1 int i;
      /* Use file position and file size as part of the key. */
      register s_int16 rnd_seed =
	(s_int16)((s_int32)(filePos ^ cs->fileSize)>>9)
	^ (s_int16)cs->fileSize ^ FIRMWARE_KEY;
      register u_int16 *p = &tmp[0];
      for (i=0;i<256;i++) {
	rnd_seed = rnd_seed * 16217 + 9043;
	*p++ ^= rnd_seed;
      }
    }
    MemCopyPackedBigEndian(data, firstOdd, tmp, byt, len);
    bytes -= len;
    firstOdd += len;
    data += firstOdd/2;
    firstOdd &= 1;
  }
  return read;
}

u_int16 DecryptPlayCurrentFile(void) {
    u_int32 tmp;

    LoadCheck(NULL, 0); /* higher clock, but 4.0x not absolutely required */

    /* 1) Check for OggS, use DecryptReadDiskSector if needed. */
    ReadFile((void *)&tmp, 0, 4);
    Seek(0);
    /* if not "OggS" */
    if (tmp != 0x67534f67 /*gSOg*/) {
      putstrp("\pTrying Decrypting\n");
#if 0
	/* Forces re-read and decryption of the first sector */
	minifatInfo.currentSector = -1;
	/*oldVect =*/
	SetHookFunction((u_int16)ReadDiskSector, DecryptReadDiskSector);
	/* 2) Play file */
	RealPlayCurrentFile();

	/* 3) Restore old ReadDiskSector */
	SetHookFunction((u_int16)ReadDiskSector,
			MyReadDiskSector/*oldVect*/);

#else
	{
	  void *oldRd = cs.Read;
	  cs.Read = DecryptCsRead;
	  RealPlayCurrentFile();
	  cs.Read = oldRd;
	}
#endif
      putstrp("\pDone Decrypting\n");
    } else {
	/* 2) Play file */
	RealPlayCurrentFile();
    }
    return 0;
}
#endif /*USE_ENCRYPTION*/

#ifdef GAPLESS
enum CodecError PlayGapless(void) {
    u_int16 ret = ceOtherError;
    const char *eStr;
    /*If you use fixed clock (with UART), you can comment out the LoadChecks.*/
    LoadCheck(NULL, 0); /* higher clock -- if not already */
    if (codecVorbis.audioBegins) {
	u_int16 data[2];
	Seek(codecVorbis.audioBegins);
	ReadFile(data, 0, 4);
	Seek(0); //Seek(0) keeps original fileLeft correct
	if (data[0] == 0x4f67 && data[1] == 0x6753) {
	    /* 'OggS', assume the same parameters,
	       skip Vorbis decoder init. */
	    OggSeekHeader(codecVorbis.audioBegins);
	    ogg[1]/*.headerTypeFlag*/ = 0;
	    ogg[6]/*.streamSerialNumber*/ = 0;
	    ogg[7]/*.streamSerialNumber*/ = 0;
	    ogg[14]/*.cont*/ = 0;
	    vFirstFrame = 1;
	    cs.playTimeSeconds = 0;
	    cs.playTimeSamples = 0;
	    /*Start decoding without recreating the decoding tables
	      from header.*/
putch('g');
	} else {
	    /* 'OggS' Not found, must be a file with different parameters
	       (or WAV) */
	normal:
#ifdef USE_WAV
	    //CodMicroWavCreate(); /*We know it does nothing */
	    /* call Decode directly to save some instructions */
	    ret = CodMicroWavDecode(cod, &cs, &eStr);
	    //CodMicroWavDelete(cod); /*We know it does nothing */
	    if (ret != ceFormatNotFound) {
#ifdef LOOP_MODE
		leaveLoop = 0;
		while (loopMode && !leaveLoop && cs.fileLeft < 5/*arbitrary end limit*/) {
//putch('l');
		    LoadCheck(NULL, 0); /*higher clock*/
		    cs.Seek(&cs, 0, SEEK_SET); /* seek to the beginning */
		    CodMicroWavDecode(cod, &cs, &eStr); /* and decode again */
#ifdef GAPLESS
		    /* Must clear this after using the vorbis area! */
		    codecVorbis.audioBegins = 0;
#endif/*GAPLESS*/
		    IdleHook(); //so that it is called sometime!
		}
#endif/*LOOP_MODE*/
		return ret;
	    }
	    /* If failed, seek to the beginning and try Ogg Vorbis decoding. */
	    cs.Seek(&cs, 0, SEEK_SET);
#endif/*USE_WAV*/
	    if (!(cod = CodVorbisCreate())) {
		return ceOtherError; //should never happen
	    }
putch('-');
	}
	/* Decode the file once */
#ifdef HAS_PATCHCODVORBISDECODE
	/*available in latest dev1000.h */
	ret = PatchCodVorbisDecode(&codecVorbis, &cs, &eStr, 0); 
#else
	ret = CodVorbisDecode(&codecVorbis, &cs, &eStr); 
#endif
#ifdef LOOP_MODE
	/* If loop mode, and played the file to the end.
	   Clearing loop mode in the middle will play a repeat to the end,
	   then return from play.
	 */
	leaveLoop = 0;
	while (loopMode && !leaveLoop && cs.fileLeft < 5/*arbitrary end limit*/) {
//putch('l');
	    LoadCheck(NULL, 0); /*higher clock*/
	    /* Perform the minimal clearing */
	    ogg[1]/*headerTypeFlag*/ = 0;
	    CodVorbisGoTo(cod, 0);

	    ogg[1]/*.headerTypeFlag*/ = 0;
	    ogg[14]/*.cont*/ = 0;
	    vFirstFrame = 1; /* skip the first windowed samples */

#ifdef HAS_PATCHCODVORBISDECODE
	    /*available in latest dev1000.h */
	    ret = PatchCodVorbisDecode(&codecVorbis, &cs, &eStr, 0); 
#else
	    ret = CodVorbisDecode(&codecVorbis, &cs, &eStr); 
#endif
	    IdleHook(); //so that it is called sometime!
	}
#endif /*LOOP_MODE*/
	//cod->Delete(cod); /* does not do anything */
    } else {
	goto normal;
    }
    return ret;
}
#endif/*GAPLESS*/


#if defined(USE_WAV) && !defined(GAPLESS)
#include <codecmicrowav.h>
#include <dev1000.h>
extern u_int16 codecVorbis[];
extern u_int16 ogg[];
extern u_int16 mInt[];

extern struct CodecServices cs;
extern struct Codec *cod;
enum CodecError PlayWavOrOggFile(void) {
  register enum CodecError ret = ceFormatNotFound;
  const char *eStr;
  LoadCheck(NULL, 0); /* higher clock, but 4.0x not absolutely required */
  /* We know create does nothing */
#ifdef USE_WAVGOTO
  ret = CodGotoWavDecode(cod, &cs, &eStr);
#else
  ret = CodMicroWavDecode(cod, &cs, &eStr);
#endif
#if 0
  {
    register int i;
    for (i=0;i<DEFAULT_AUDIO_BUFFER_SAMPLES/(sizeof(tmpBuf)/2);i++) {
      /* Must be memset each time, because effects like EarSpeaker are
	 performed in-place in the buffer */
      memset(tmpBuf, 0, sizeof(tmpBuf));
      AudioOutputSamples(tmpBuf, sizeof(tmpBuf)/2);
    }
  }
#endif
  if (ret == ceFormatNotSupported)
    return ceFormatNotFound; /* makes prev work with unsupported files */
  if (ret != ceFormatNotFound)
    return ret;
  cs.Seek(&cs, 0, SEEK_SET);
  return PatchPlayCurrentFile();
}
#endif /*!GAPLESS && USE_WAV*/
  

#ifdef USE_USB
#if defined(PATCH_EJECT) || defined(PATCH_UNIT_READY) || defined(PATCH_SENSE)
#include <scsi.h>
auto u_int16 NewDiskProtocolCommand(register __i2 u_int16 *cmd) {
    __y u_int16 c = (cmd[OPERATION_CODE] & 0xff);
#if defined(PATCH_EJECT)
    if (c == SCSI_START_STOP_UNIT) {
	/* 0:OP 1:imm0 2:res 3:res 4:power7-4 res3-2 loej1 start0 5:control */
	startUnit = cmd[4] & 1;
    } else if (startUnit == 0) {
	/* The next command after STOP UNIT will not be serviced
	   fully because we will turn ourselves off. */
	startUnit = 2;
    }
#endif
#if defined(PATCH_UNIT_READY)
    if (c == SCSI_TEST_UNIT_READY) {
	/* Poll MMC present by giving it a command.
	   But only when idle, i.e. mmcOk. */
	if (mmc.state == mmcOk && mmc.errors == 0 &&
	    MmcCommand(MMC_SET_BLOCKLEN|0x40, 512) != 0) {
	    mmc.errors++;
	}
	if (mmc.state == mmcNA || mmc.errors) {
	    SCSI.Status = SCSI_REQUEST_ERROR; /* report error at least once! */
	}
	SCSI.State = SCSI_SEND_STATUS;
	return -1; /*done processing*/
    }
#endif
#if defined(PATCH_SENSE)
    if (c == SCSI_REQUEST_SENSE) {
	/* patch a bug -- sense was written to the wrong buffer */
	SCSI.DataBuffer[0] = 0x7000;
	SCSI.DataOutBuffer = SCSI.DataBuffer;
	SCSI.DataOutSize = 18; /*Windows requires 18.*/
	SCSI.State = SCSI_DATA_TO_HOST;
	return -1; /* done with the command */
    }
#endif
    return cmd[OPERATION_CODE]; /* perform the command */
}
#endif
#endif /*USE_USB*/



 



/*
  put packed string
 */
void putstrp(register __i0 u_int16 *packedStr) {
#ifndef GAPLESS
    while (1) {
	register int i = *packedStr++;
	if (i >> 8) {
	    putch(i >> 8);
	} else {
	    break;
	}
	if (!(i & 255)) {
	    break;
	}
	putch(i);
    }
#endif
}




/* Our own loadcheck which will use fixed 3.0x clock for player
   and 4.0x for USB. But as we don't have USB mode in this program,
   we would only need fixed clock. */
void UartLoadCheck(struct CodecServices *cs, s_int16 n) {
  if (cs == NULL) {
#if 1 /* USB is not serviced in spiplayer, but is in sdplayer. */
    if (n) {
	clockX = 8; /* USB requires 4.0x! */
	goto set;
    }
#endif
    if (clockX != player.maxClock) {
	clockX = player.maxClock; /* Otherwise 3.0-3.5x */
    set:
	SetRate(hwSampleRate);
    }
    return;
  }
  /* cs is always non-NULL here */
  if (cs->gain != player.volumeOffset) {
    player.volumeOffset = cs->gain; /* vorbis-gain */
    PlayerVolume();
  }
}


u_int16 lineBuffer[80];
u_int16 lineCnt = 0;

/*
  Commands that do not require newline, if given at the start of the line:
  'f' - file mode (wait for play command or queued song)
  'c' - continuous play mode (play next file automatically)
  'C' - cancel current song
  '+' - volume up
  '-' - volume down
  '' - faster play
  '=' - pause
  '>' - play (normal speed)
  '?' - inquiry position
  ',' - previous file
  '.' - next file
  'R' - reset play queue, also cancels the currently playing song

  Commands that require a newline:
  "p<num>\n" - play file <num>
  "P<name>\n" - play file <name> (for example "PAUDIO_00OGG\n")
  "OFF\n" - turn unit off
  "L\n" - list files
  "q<num>\n" - queue file <num>
  "Q<name>\n" - queue file <name>
  "v<vol>\n" - set volume to <vol>
 */
#include <minifat.h>
struct FRAGMENT minifatFragmentsSave[MAX_FRAGMENTS];

void UartIdleHook(void) {
    if (UartFill()) {
	register int cmd = UartGetByte();
//putch(cmd);
	if (lineCnt == 0) {
	    if (cmd == 'f') {
		/* file mode */
		playerMode = PLAYER_WAIT_FOR_COMMAND;
		player.nextFile = -1;
		goto echo;
	    } else if (cmd == 'c') {
		/* continuous play mode */
		playerMode = PLAYER_CONTINUOUS_PLAY;
		player.nextFile = player.currentFile + 1;
		goto echo;
	    } else if (cmd == 'C') {
		putch('C');
		goto cancelpauseoff;
		//cs.cancel = 1;
		//player.pauseOn = 0;
	    } else if (cmd == '+') {
		KeyEventHandler(ke_volumeUp2);
		goto showvol;//putword(player.volume);
	    } else if (cmd == '-') {
		KeyEventHandler(ke_volumeDown2);
	    showvol:
		putword(player.volume);
	    } else if (cmd == '=') {
		/* pause */
		player.pauseOn = 1;
		goto echo;//putch('=');
	    } else if (cmd == '>') {
		/* play */
		player.pauseOn = 0;
		cs.fastForward = 1;
		goto echo;//putch('>');
	    } else if (cmd == '') {
		/* Fast forward */
		//if (cs.fastForward < 4)
		++cs.fastForward;
		putch(cs.fastForward);
	    } else if (cmd == '.') {
		/* next */
		player.nextFile = player.currentFile + 1;
		putch('n');
		goto cancelpauseoff;
		//cs.cancel = 1;
		//player.pauseOn = 0;
	    } else if (cmd == ',') {
		/* previous */
		if (cs.playTimeSeconds >= 5) {
		    player.nextFile = player.currentFile;
		} else if (player.currentFile) {
		    player.nextFile = player.currentFile - 1;
		} else {
		    player.nextFile = player.totalFiles-1;
		}
		putch('p');
		goto cancelpauseoff;
		//cs.cancel = 1;
		//player.pauseOn = 0;
	    } else if (cmd == '?') {
		/* position + other info -- Note: will block. */
		putword(cs.playTimeSeconds>>16);
		putword(cs.playTimeSeconds);
		putch(cs.fileLeft / ((cs.fileSize >> 8)+1));
#ifdef QUEUE_SIZE
	    } else if (cmd == 'R') {
		queued = 0;
		putch('R');
		goto cancelpauseoff;
#endif
#ifdef LOOP_MODE
	    } else if (cmd == 'l') {
		loopMode = 1-loopMode;
		putch(loopMode + '0');
#endif
	    } else {
		/*unknown command*/
		if (cmd != '\n' && cmd != '\r') {
		    lineBuffer[lineCnt++] = cmd;
		} else {
		    lineCnt = 0;
		}
	    echo:
		putch(cmd);
	    }
	} else {
	    if (cmd != '\n' && cmd != '\r' && lineCnt < sizeof(lineBuffer)) {
		lineBuffer[lineCnt++] = cmd;
	    } else {
		u_int16 fName[10/2];
		__y u_int32 fSuffix =
		    FAT_MKID(lineBuffer[9],lineBuffer[10],lineBuffer[11]);
				/* Build a 8-character filename */
		fName[0] = (lineBuffer[1]<<8) | lineBuffer[2];
		fName[1] = (lineBuffer[3]<<8) | lineBuffer[4];
		fName[2] = (lineBuffer[5]<<8) | lineBuffer[6];
		fName[3] = (lineBuffer[7]<<8) | lineBuffer[8];
		fName[4] = ('\n'<<8) | 0;

		lineBuffer[lineCnt++] = '\0';
		lineCnt = 0;
		if (!memcmp(lineBuffer, "OFF", 3)) {
		    putch('o');
		    RealPowerOff();
		}
		switch (lineBuffer[0]) {
#ifdef USE_WAVGOTO
		case 'g':
		  cs.goTo = atoul(lineBuffer+1) / 1000; //for Ogg Vorbis
		  longGoTo = atoul(lineBuffer+1);
		  break;
#endif
		case 'L':
		    /* List files */
		    //minifatInfo.supportedSuffixes = NULL;
		    /*all files*/
		{
		    /* Save FAT state */
		    u_int32 fileSize = minifatInfo.fileSize;
		    __y u_int32 filePos = minifatInfo.filePos;
		    memcpyYX(&minifatFragmentsSave, &minifatFragments,
			     sizeof(minifatFragmentsSave));

		    IterateFiles();

		    /* Restore FAT state */
		    memcpyXY(&minifatFragments, &minifatFragmentsSave,
			     sizeof(minifatFragments));
		    minifatInfo.fileSize = fileSize;
		    minifatInfo.filePos = filePos;
		    break;
		}

#ifdef QUEUE_SIZE
		case 'q':
		    if (queued < QUEUE_SIZE) {
			u_int16 num = atou((void *)&lineBuffer[1]);
			if (num < player.totalFiles) {
			    queue[queued++] = num;
			    putstrp("\padd ");
			    putch('0'+num/10);
			    putch('0'+num%10);
			    putch('\n');
			} else {
			    putstrp("\pnof\n");
			}
		    } else {
			putstrp("\pful\n");
		    }
		    break;
#endif/*QUEUE_SIZE*/

#ifdef QUEUE_SIZE
		case 'Q':
		    if (queued < QUEUE_SIZE) {
			u_int16 num;
			{
			    /* Save FAT state */
			    u_int32 fileSize = minifatInfo.fileSize;
			    __y u_int32 filePos = minifatInfo.filePos;
			    memcpyYX(&minifatFragmentsSave, &minifatFragments,
				     sizeof(minifatFragmentsSave));

			    num = OpenFileNamed(fName, fSuffix);

			    /* Restore FAT state */
			    memcpyXY(&minifatFragments, &minifatFragmentsSave,
				     sizeof(minifatFragments));
			    minifatInfo.fileSize = fileSize;
			    minifatInfo.filePos = filePos;
			}

			if (num != 0xffffU) {
			    queue[queued++] = num;
			    putstrp("\padd ");
			    putch('0'+num/10);
			    putch('0'+num%10);
			    putch(' ');
			} else {
			    putstrp("\pnof ");
			}
			putstrp(fName);
		    } else {
			putstrp("\pful\n");
		    }
		    break;
#endif /*QUEUE_SIZE*/

		case 'P':  /* Play by name */
		{
		    /* Save FAT state */
		    u_int32 fileSize = minifatInfo.fileSize;
		    __y u_int32 filePos = minifatInfo.filePos;
		    memcpyYX(&minifatFragmentsSave, &minifatFragments,
			     sizeof(minifatFragmentsSave));

		    player.nextFile = OpenFileNamed(fName, fSuffix);

		    /* Restore FAT state */
		    memcpyXY(&minifatFragments, &minifatFragmentsSave,
			     sizeof(minifatFragments));
		    minifatInfo.fileSize = fileSize;
		    minifatInfo.filePos = filePos;
		    goto checkvalidfile;
		}

		case 'p': /* Play by number */
		    player.nextFile = atou((void *)&lineBuffer[1]);
		checkvalidfile:
		    if (player.nextFile >= player.totalFiles) {
			player.nextFile = -1;
		    }
		    goto cancelpauseoff;
		    //break;

		case 'v': /* Set volume */
		    player.volume = atou((void *)&lineBuffer[1]) - 12;
		    PlayerVolume();
		    goto showvol;
		}
	    }
	}
    }

    if (uiTrigger) {
	uiTrigger = 0;

#if 0   /* This version does not allow keys to be used. */
	KeyScanNoUSB(0x1f); /* with feature button */
#endif
	//CheckSd();
	if (USBIsAttached()) {
	cancelpauseoff:
	    cs.cancel = 1;
	    player.pauseOn = 0;
	    player.currentFile = -1;
	}
    }
#if defined(LOOP_MODE) || defined(QUIET_CANCEL)
    if (cs.cancel) {
#ifdef LOOP_MODE
	leaveLoop = 1;
#endif
#ifdef QUIET_CANCEL
	volumeReg = 0xc0c0;
	SetVolume();
#endif
    }
#endif/*defined(LOOP_MODE) || defined(QUIET_CANCEL)*/
}



void UartInit(void) {
    Disable();
    uartRxWrPtr = uartRxRdPtr = uartRxBuffer;
    WriteIRam(0x20+INTV_RX, ReadIRam((u_int16)MyRxIntCommand));
    //PERIP(INT_ENABLEL) |= INTF_RX;
    Enable();
#if 1
    /*read out any remaining chars*/
    while (UartFill()) {
	UartGetByte();
    }
#endif
}






#if 0
void CheckSd(void) {
    if ((PERIP(GPIO0_IDATA) & (1<<14))) {
	/* SD not inserted */
	putstrp("\p!SD\n");

	/* If SD is not inserted, run SPI FLASH player. */
#ifndef USE_USB
	//SetHookFunction((u_int16)ReadDiskSector, MapperReadDiskSector);
#endif
	PERIP(SPI0_CONFIG) = SPI_CF_MASTER | SPI_CF_DLEN8 | SPI_CF_FSIDLE1;
	ResetIRAMVectors();
	SpiLoad(SPIPLAYERSTART+4, 1/*24-bit address*/);
    }
}
#else
#define CheckSd() if((PERIP(GPIO0_IDATA) & (1<<14)))break
#endif


void IterateFilesCallback(register __b0 u_int16 *name) {
    register int i;
    /* Dump the whole FAT directory entry */
    for (i=0;i<32/2;i++) {
	putword(*name++);
    }
#ifdef SHOW_LONG_FILENAMES
    if (minifatInfo.longFileName[0]) {
	register __y s_int16 *p = minifatInfo.longFileName;
	while (*p >> 8) {
	    putch(*p >> 8);
	    if (*p & 255) {
		putch(*p & 255);
	    } else {
		break;
	    }
	    p++;
	}
    }
    putch('\n');
#endif
}

#define GPIO0_SD_POWER GPIO0_ALE
extern u_int16 underflowCount;
extern u_int16 rateCheckCount;
void main(void) {
    Enable();

#ifdef GAPLESS
    memset(&codecVorbis, 0, sizeof(codecVorbis));
    //codecVorbis.audioBegins = 0;
#endif
#if 1
    clockX = 2;
    InitAudio(); /* goto 3.0x..4.0x */
    PERIP(INT_ENABLEL) = INTF_RX | INTF_TIM0;
    PERIP(INT_ENABLEH) = INTF_DAC;
#endif
    clockX = player.maxClock;
    SetRate(cs.sampleRate=8000);
    USEX(UART_DIV) = UartDiv();
#ifdef USE_WAVGOTO
    putstrp("\pSDPlayerGoto\n");
#endif
    
#ifdef USE_WAV
    /* allow both .WAV and .OGG */
#if 0 //def GAPLESS
    { /*for testing, play .WAV only*/
	static const u_int32 sup[] = {FAT_MKID('W','A','V'), 0};
	supportedFiles = sup;
    }
#else
    supportedFiles = defSupportedFiles; /* WAV OGG */
#endif
#endif

#if 0
    { /*for testing */
	static const u_int32 sup[] = {FAT_MKID('O','G','E'),
				      FAT_MKID('B','1',' '),
				      FAT_MKID('C','1',' '),
				      0};
	supportedFiles = sup;
    }
#endif
    

#if !defined(USE_WAV) || !defined(USE_USB)
    /* Ignore subdirectories in FAT12 disks */
    SetHookFunction((u_int16)OpenFile, Fat12OpenFile);
#endif
    //voltages[voltCorePlayer] = voltages[voltCoreUSB] = 29;//27;
    //voltages[voltIoUSB] = voltages[voltIoPlayer] = 27; /* 3.3V */
    //voltages[voltAnaPlayer] = 30; /*3.6V for high-current MMC/SD!*/
    //PowerSetVoltages(&voltages[voltCorePlayer]);

    //SetHookFunction((u_int16)IdleHook, NullHook);   /* no default keyscan */
    SetHookFunction((u_int16)IdleHook, UartIdleHook);
    SetHookFunction((u_int16)USBSuspend, NullHook);   /* no low-power state */
    SetHookFunction((u_int16)LoadCheck, UartLoadCheck); /* fixed clock */

#ifdef USE_USB
    /*fixes SCSI_READ_10 residue problem -- mainly for Linux compatibility,
      does not fix LBAB, but is shorter.. */
    SetHookFunction((u_int16)MSCPacketFromPC, PatchDiskProtocolCommandDE);
#endif/*USE_USB*/

    /* Replicate main loop */
//    player.volume = 17;
//    player.volumeOffset = -24;
    player.pauseOn = 0;
    //player.randomOn = 0;
//    keyOld = KEY_POWER;
//    keyOldTime = -32767; /* ignores the first release of KEY_POWER */

    /*
      MMC mapper replaces FLASH mapper, so we do not need to hook
      ReadDiskSector.

      We must also replace the MassStorage() functionality, because it
      automatically deletes mapper, creates a read-write FLASH mapper,
      and restores read-only FLASH mapper at exit.

      We need to replace the main loop because we want to detect
      MMC/SD insertion and removal.

      We do not hook the MassStorage, because we can call our own version
      from the main loop without using the RAM hook. This saves some
      instruction words.
     */
#if 0
    map = FsMapMmcCreate(NULL, 0); /* Create MMC Mapper */
#else
    /* inlined */
    PERIP(GPIO0_MODE) &= ~(MMC_MISO|MMC_CLK|MMC_MOSI|MMC_XCS);
    PERIP(GPIO0_DDR) = (PERIP(GPIO0_DDR) & ~(MMC_MISO))
	| (MMC_CLK|MMC_MOSI|MMC_XCS)
	| GPIO0_SD_POWER;
    //PERIP(GPIO0_ODATA) = (PERIP(GPIO0_ODATA) & ~(MMC_CLK|MMC_MOSI))
    //| (MMC_XCS | GPIO0_CS1); /* NFCE high */
    PERIP(GPIO0_SET_MASK)   = (MMC_XCS | GPIO0_CS1); /* NFCE high */
    PERIP(GPIO0_CLEAR_MASK) = (MMC_CLK|MMC_MOSI);
    memset(&mmc, 0, sizeof(mmc));
#ifdef USE_USB
    map = &mmcMapper;
#else
    SetHookFunction((u_int16)ReadDiskSector, MyReadDiskSector);
#endif
#endif


//    player.volume = -24; /* "max" volume */
    //PlayerVolume();


    UartInit(); //Set interrupt etc.

//    PERIP(SCI_STATUS) &= ~SCISTF_ANADRV_PDOWN;

    while (1) {
	//CheckSd();

	/* If MMC is not available, try to reinitialize it. */
	if (mmc.state == mmcNA || mmc.errors) {
	    /* enable SD power */
	    PERIP(GPIO0_SET_MASK) = GPIO0_SD_POWER;
	    InitializeMmc(50);
	}

#ifdef USE_USB
	/* If USB is attached, call our version of the MassStorage()
	   handling function. */
	if (USBIsAttached()) {
	    putstrp("\pUSB Attach\n\n");
	    MyMassStorage();
	    putstrp("\pUSB Detach\n");
#ifdef GAPLESS
	    /* Must clear this after using the vorbis area! */
	    codecVorbis.audioBegins = 0;
#endif
	}
#endif /*USE_USB*/

	/* Try to init FAT. */
	if (InitFileSystem() == 0) {
	    /*
	      Check if there is an update file (SDUPDATE.PRG) on the SD card.
	    */
	    /* If OpenFileNamed() is used anyway, use this version */
	    if (OpenFileNamed("\pSDUPDATE", FAT_MKID('P','R','G'))!=0xffffU) {
		/* Note:
		   The maximum size for the program image is 8192 bytes.
		*/
		if (ReadFile(mallocAreaX, 0, 0x2000)) {
		    Disable();
		    BootFromX(mallocAreaX+8); // Exit to new image     
		    /* BootFromX returns only in case of invalid prg file.
		       Program execution would be unsafe after failed
		       BootFromX so restart using watchdog reset. */
#ifndef USE_WAV
		    USEX(WDOG_CONFIG) = 2;
		    USEX(WDOG_RESET) = 0x4ea9; /*Activate watchdog*/
#endif
		    while (1)
			;
		}
	    }
#if 0 //**
	    putstrp("\pfat");
	    putword(minifatInfo.totSize>>16);
	    putword(minifatInfo.totSize);
	    putch('\n');
#endif//**
	    /* Restore the default suffixes. */
	    minifatInfo.supportedSuffixes = supportedFiles;
	    player.totalFiles = OpenFile(0xffffU);
#if 0 //**
	    putstrp("\pfiles");
	    putword(player.totalFiles);
	    putch('\n');
#endif//**
	    if (player.totalFiles == 0) {
		/* If no files found, output some samples.
		   This causes the idle hook to be called, which in turn
		   scans the keys and you can turn off the unit if you like.
		*/
		putstrp("\pnofiles");
		goto noFSnorFiles;
	    }

	    /*
	      Currently starts at the first file after a card has
	      been inserted.

	      Possible improvement would be to save play position
	      to SPI EEPROM for a couple of the last-seen MMC/SD's.
	      Could also save the play mode, volume, and earspeaker
	      state to EEPROM.
	    */
	    player.nextStep = 1;
	    if (playerMode == PLAYER_CONTINUOUS_PLAY) {
		player.nextFile = 0;
	    } else {
		player.nextFile = -1;
	    }
	    player.currentFile = -1;

#if 0 //**DEBUG
	    PERIP(SCI_STATUS) &= ~SCISTF_ANADRV_PDOWN;
	    player.nextFile = 1;
	    //loopMode = 1;
	    playerMode = PLAYER_WAIT_FOR_COMMAND;
#endif

	    while (1) {
		//CheckSd();
		while ( player.nextFile == -1
#ifdef QUEUE_SIZE
			&& queued == 0
#endif
		    ) {
		    memset(tmpBuf, 0, sizeof(tmpBuf)); /* silence */
		    AudioOutputSamples(tmpBuf, sizeof(tmpBuf)/2); /*no pause*/
		    //cs.Output(&cs, tmpBuf, sizeof(tmpBuf)/2);
		    //CheckSd();
		}
		/* If the file can be opened, start playing it. */

#ifdef QUEUE_SIZE
		if (queued) {
		    putstrp("\pqueue ");
		    putch(queued/10+'0');
		    putch(queued%10+'0');
		    putch('\n');
		    queued--;
		    player.currentFile = queue[0];
		    memmove(&queue[0], &queue[1], QUEUE_SIZE-1);
		    player.nextFile = -1;
		} else
#endif/*QUEUE_SIZE*/
		{
		    player.currentFile = player.nextFile;
		    if (playerMode == PLAYER_CONTINUOUS_PLAY) {
			player.nextFile = player.currentFile + 1;
		    } else {
			player.nextFile = -1;
		    }
		}

		if (OpenFile(player.currentFile) < 0) {
//putch('o');
		    //player.volumeOffset = -12; /*default volume offset*/
		    //PlayerVolume();

		    player.ffCount = 0;
		    cs.cancel = 0;
		    cs.goTo = -1; /* start playing from the start */
		    cs.fileSize = cs.fileLeft = minifatInfo.fileSize;
		    cs.fastForward = 1; /* reset play speed to normal */

#if !defined(GAPLESS)
		    putstrp("\pplay");
		    putword(player.currentFile);

		    putword(minifatInfo.fileName[0]);
		    putword(minifatInfo.fileName[1]);
		    putword(minifatInfo.fileName[2]);
		    putword(minifatInfo.fileName[3]);
		    putword(minifatInfo.fileName[4]);
		    putch(minifatInfo.fileName[5]>>8);
		    putch('\n');
#endif

		    
		    PlayerVolume(); /*required for QUIET_CANCEL*/
#ifdef USE_ENCRYPTION
		    DecryptPlayCurrentFile();
#else

#ifdef GAPLESS
		    PlayGapless();
#else/*GAPLESS*/
#ifdef USE_WAV
		    PlayWavOrOggFile();
#else
		    PatchPlayCurrentFile();
#endif
#endif/*else GAPLESS*/
#endif/*USE_ENCRYPTION*/
		    player.currentFile = -1;
#if !defined(GAPLESS)
		    putstrp("\pdone\n");
#endif
		} else {
//putch('x');
		    player.currentFile = -1;
		    player.nextFile = 0;
		}
		/* Leaves play loop when MMC changed */
		if (mmc.state == mmcNA || mmc.errors) {
#if !defined(GAPLESS)
		    putstrp("\pdone\n");
#endif
		    break;
		}
#ifdef USE_USB
		if (USBIsAttached()) { /* .. or USB is attached. */
		    break;
		}
#endif
	    }
	} else {
	    /* If not a valid FAT (perhaps because MMC/SD is not inserted),
	       just send some samples to audio buffer and try again. */
	    putstrp("\pnofat\n");
	noFSnorFiles:
#ifdef GAPLESS
	    codecVorbis.audioBegins = 0;
#endif
	    //if (AudioBufFill() < sizeof(tmpBuf)/2) { /*Minimize delay*/
	    LoadCheck(&cs, 32); /* decrease or increase clock */
	    memset(tmpBuf, 0, sizeof(tmpBuf));
	    AudioOutputSamples(tmpBuf, sizeof(tmpBuf)/2);
	    /* When no samples fit, calls the user interface
	       -- handles volume control and power-off. */
	    //} else {
	    //IdleHook();
	    //}
	}
    }
}
