#include <stdio.h>
#include <strings.h>
#include <unistd.h>
#include <errno.h>
#include <termios.h>
#include <fcntl.h>
#include <time.h>
#include <stdlib.h>

#define STATUS_SIZE 16

#if 0
#define HIGH_SPEED_RS232
#endif

#ifdef HIGH_SPEED_RS232
#define RS232_SPEED 115200
#else
#define RS232_SPEED 9600
#endif

static int newsfd;
static struct termios oldtio;
static char commDevice[]="/dev/cuaa1";
static char *symNames[STATUS_SIZE] = {
  "ST_INIT",
  "ST_MP3_MEMTEST",
  "ST_MMC_BLOCKSH",
  "ST_MMC_BLOCKS2",
  "ST_MMC_BLOCKS1",
  "ST_MMC_BLOCKS0",
};
 
static int InitComm(int initBaudRate) {
  struct termios newtio;

  /* open the device to be non-blocking (read will return immediately) */
  newsfd = open(commDevice, O_RDWR | O_NOCTTY | O_NONBLOCK);
  if (newsfd <0) {perror(commDevice); exit(-1); }
  
  /* allow the process to receive SIGIO */
  fcntl(newsfd, F_SETOWN, getpid());
  /* Make the file descriptor asynchronous (the manual page says only
     O_APPEND and O_NONBLOCK, will work with F_SETFL...) */
  fcntl(newsfd, F_SETFL, O_ASYNC);
  
  tcgetattr(newsfd,&oldtio); /* save current port settings */
  /* set new port settings for canonical input processing */
  bzero(&newtio,sizeof(newtio));
  newtio.c_cflag = CS8 | CLOCAL | CREAD ; 
  newtio.c_iflag = IGNPAR;
  newtio.c_oflag = 0;
  newtio.c_lflag = 0;
  newtio.c_cc[VMIN] = 0;
  newtio.c_cc[VTIME] = 0;
  newtio.c_ispeed = initBaudRate;
  newtio.c_ospeed = initBaudRate;
  tcflush(newsfd, TCIFLUSH);
  if (tcsetattr(newsfd,TCSANOW,&newtio)==-1) {
    printf("Serial port initialization failed, errno: %d\n",errno);
    return -1;
  }

  return 0;
}

static int SendChar(char d) {
  if (write(newsfd, &d, 1)<0)
    return -1;
  else
    return d;
}


#if 0
static int SendData(char *data, int len) {
  if (write(newsfd, data, len)<0) 
    printf("Write error\n"); 
  return 0;
}
#endif

static int ReceiveChar(void) {
    unsigned char d;
  if (read(newsfd, &d, 1) == 1) {
    return d & 0xff;
  } else {
    return -1;
  }
}

static int ReceiveData(char *data, int len) { 
  int i = 0;
  int startTime = time(NULL);

  while(i<len && time(NULL) - startTime < 2) { 
    if(read(newsfd,data+i,1)==1)
      i++;       
  }
  return i;
}

#if 0
static int CloseComm(void)  { 
  /* restore old port settings */
  tcsetattr(newsfd,TCSANOW,&oldtio);
  /* close serial port */
  close(newsfd); 
  return 0; 
}
#endif


#define MP3_BUF_SIZE (62720L*512)

static unsigned char mp3Buf[MP3_BUF_SIZE];
static int startTab[128] = {0};

int FillMp3(int bytes, int blockSize) {
  int i=0, currPos = blockSize, currLen;
  FILE *fp;
  char *fileNames[] = {
#if 1
    "Mp3LQDemo/Jussi69Eyes.mp3",
    "Mp3LQDemo/FeelIt.mp3",
    "Mp3LQDemo/HampsterDance.mp3",
    "Mp3LQDemo/Paranoid.mp3",
    "Mp3LQDemo/Popcorn.mp3",
    "Mp3LQDemo/Vesireitit.mp3",
#elif 0
    "../../hphome/proj/vsmpg/src/mp3/Short_s_44_256.mp3",
    "../../hphome/proj/vsmpg/src/mp3/Overflow_m_44_064.mp3",
    "../../hphome/proj/vsmpg/src/mp3/Test22.mp3",
    "Spin.mp3",
#else
    "../../Mp3/Misc1/HampsterDance.mp3",
#endif
    NULL
  };

  printf("Filling MP3 buffer, bytes=%d, blockSize=%d\n", bytes, blockSize);
  while (fileNames[i] && currPos < bytes) {
    if (!(fp = fopen(fileNames[i], "rb"))) {
      printf("Couldn't open %s\n", fileNames[i]);
      exit(EXIT_FAILURE);
    } else {
      printf("    Reading %s:\n", fileNames[i]);
      printf("\tArea=%8d ..", currPos); fflush(stdout);
      currLen = fread(mp3Buf+currPos, 1, bytes-currPos, fp);
      if (currLen < 0) {
	printf("file error\n");
	exit(EXIT_FAILURE);
      }
      startTab[2+2*i] = currPos;
      startTab[3+2*i] = currLen;
      currPos += currLen;
      printf("%8d (%8d bytes)\n", currPos, currLen);
      fclose(fp);
    }
    i++;
  }
  printf("Organizing...\n");
  startTab[0] = i;

  for (i=0; i<blockSize/4; i++) {
    mp3Buf[i*4  ] = startTab[i] >> 24;
    mp3Buf[i*4+1] = startTab[i] >> 16;
    mp3Buf[i*4+2] = startTab[i] >> 8;
    mp3Buf[i*4+3] = startTab[i];
  }
#if 0
  for (i=0; i<blockSize; i++) {
    printf("%3u%c", mp3Buf[i], (i+1&7) ? '\t' : '\n');
  }
#endif
  printf("MP3 buffer ok, filled %d of %d bytes (%4.2f%%)\n",
	 currPos, bytes, currPos*100.0/bytes);

  return (currPos+blockSize-1) / blockSize;
}

void c2s(char *c, int n) {
  int i;
  for (i=7; i>=0; i--)
    *c++ = '0' + ((n>>i)&1);
  *c = '\0';
}

int main(int argc, char **argv) {
  time_t lastOpT;
  char t[STATUS_SIZE] = {0};
  int i, c;
  int blockSize, blocks, usedBlocks;
  int totErr = 0;
  int mode = 0;

  if (InitComm(RS232_SPEED))
    exit(-1);

  for (i=1; i<argc; i++) {
    if (!strcmp(argv[i], "-w"))
      mode = 2;
    else if (!strcmp(argv[i], "-r"))
      mode = 1;
    else if (!strcmp(argv[i], "-n"))
      mode = 0;
    else if (!strcmp(argv[i], "-t"))
      mode = 3;
  }

  if (mode) {
    lastOpT = time(NULL);
    SendChar(4);
    while((c = ReceiveChar()) && time(NULL)-lastOpT < 2)
      ;
    if (c) {
      printf("Connection not established, err=%d!\n", c);
      exit(EXIT_FAILURE);
    }

    SendChar(0);
    while((c = ReceiveChar()) < 0 && time(NULL)-lastOpT < 2)
      ;
    if (c < 0) {
      printf("Timeout asking basic info!\n");
      exit(EXIT_FAILURE);
    } else {
      t[0] = c;
      i = ReceiveData(t+1, STATUS_SIZE-1)+1;
      printf("Received %d bytes\n", i);
      if (i<STATUS_SIZE) {
	printf("Error\n");
	exit(EXIT_FAILURE);
      }
    }
    for (i=0; i<STATUS_SIZE; i++)
      printf("t[%2d](%-14s) = %3d = 0x%02x\n",
	     i, symNames[i] ? symNames[i] : "-", t[i], t[i]);

    /* This is a kludge to correct for bad chips */
    blockSize = 1 << t[2];

    blocks = ((int)t[3]<<16) | ((int)t[4]<<8) | t[5];

    printf("Total size of device = %d bytes = %5.2f MB (%d byte blocks)\n",
	   blocks*blockSize, blocks*blockSize/(1024*1024.0), blockSize);

    usedBlocks = FillMp3(blocks*blockSize, blockSize);
    printf("%d blocks used of %d\n", usedBlocks, blocks);
    if (mode > 2) {
      SendChar(3);
      lastOpT = time(NULL);
      while((c = ReceiveChar()) && time(NULL)-lastOpT < 2)
	;
      if (c) {
	printf("Couldn't return to user interface, err=%d!\n", c);
	exit(EXIT_FAILURE);
      }

      exit(EXIT_SUCCESS);
    }

    if (mode == 1) {
      for (i=0; i<usedBlocks; i++) {
	printf("read block %d of %d\n", i+1, usedBlocks);
	SendChar(2);
	SendChar(i>>16);
	SendChar(i>>8);
	SendChar(i);
	lastOpT = time(NULL);
	while((c = ReceiveChar()) < 0 && time(NULL)-lastOpT < 2)
	  ;
	if (c) {
	  printf("problems: %d\n", c);
	  i--;
	} else {
	  int j;
	  for (j=0; j<blockSize; j++) {
	    lastOpT = time(NULL);
	    while((c = ReceiveChar()) < 0 && time(NULL)-lastOpT < 2)
	      ;
	    if (c<0) {
	      printf("Operation timed out, i=%d, j=%d\n", i, j);
	    }
	    if (c != mp3Buf[blockSize*i+j]) {
#if 1
	      char s1[9], s2[9];
	      c2s(s1, mp3Buf[blockSize*i+j]);
	      c2s(s2, c);
	      printf("i=%d, j=%3d: mp3Buf=%3d (%s), got=%3d (%s)\n",
		     i, j, mp3Buf[blockSize*i+j], s1, c, s2);
#endif
	      totErr++;
	    }
	  }
	  lastOpT = time(NULL);
	  while((c = ReceiveChar()) < 0 && time(NULL)-lastOpT < 2)
	    ;
	  printf("Read operation status: %d, totErr=%d\n", c, totErr);
	}
      }
    } else {	/* Mode == 2 */
      for (i=0; i<usedBlocks; i++) {
	printf("\rwrite block %d of %d", i+1, usedBlocks);
	fflush(stdout);
	SendChar(1);
	SendChar(i>>16);
	SendChar(i>>8);
	SendChar(i);
	lastOpT = time(NULL);
	while((c = ReceiveChar()) < 0 && time(NULL)-lastOpT < 2)
	  ;
	if (c) {
#if 1
	  printf("problems: %d\n", c);
#endif
	  i--;
	} else {
	  int j;
	  for (j=0; j<blockSize; j++) {
	    while (SendChar(mp3Buf[blockSize*i+j]) < 0)
	      ;
	  }
	  lastOpT = time(NULL);
	  while((c = ReceiveChar()) < 0 && time(NULL)-lastOpT < 2)
	    ;
	  printf(", write operation status: %d (%s)",
		 c, (c == 5) ? "ok" : "fail");
	  fflush(stdout);
	  if (c != 5) {
	    totErr++;
	  }
	}
      }
      printf("\nA total of %d unrecoverable errors\n", totErr);
    }
  } else {	/* Mode == 0 */
    printf("Interactive mode\n");

    fcntl(fcntl(STDIN_FILENO,  F_DUPFD, 0), F_SETFL, O_NONBLOCK);

    while(1) {
	static int val = 0;
      char d;
      int c;

      if (1 && (c = getchar()) >= 0) {
	/*	printf("ch=0x%04x\n", c);*/
	  if (c >= '0' && c <= '9') {
	      val = (val<<4) | (c - '0');
	  } else if (c >= 'A' && c <= 'F') {
	      val = (val<<4) | (c - 'A' + 10);
	  } else if (c >= 'a' && c <= 'f') {
	      val = (val<<4) | (c - 'a' + 10);
	  }
	  if (c == '\n') {
	      printf("Sent 0x%02x\n", val & 255);
	      SendChar(val & 255);
	      val = 0;
	  }
      } else if ((c = ReceiveChar()) >= 0) {
#if 1
	printf("%c %02x ", c, c);
#else
	putchar((c == '\r') ? '\n' : c);
#endif
	fflush(stdout);
      } else {
	usleep(20000);
      }
    }
  }

  return 0;
}
