/// \file standalone.c -- VS1053 Stand-Alone MP3 Player Source Code
/// Copyright (C) 2005-2009 VLSI Solution Oy Tampere Finland
// See readme.txt.

/*
  Note:

  DREQ is changed in normal firmware both in SDI interrupts when the
  stream buffer becomes too full, and whenever the stream buffer becomes
  empty enough.

  So, we can use DREQ ourselves as we wish and access to it does not need to
  be protected by interrupt Disable()/Enable(). However, the normal decoding
  will still keep raising DREQ when it sees that there is space for new data.

  VS1053: I2S and higher reference
  VS1053B has extended GPIO registers:
  GPIDATA[8](SCLK),[9](XCS),[10](SI),[11](XDCS) and GPODATA[8](SO)
*/


#define GPIO_PIN_USB_DETECT 9
#define GPIO_PIN_SDCARD_OFF 8

//#define NOCHECKFILETYPE 
//#define USE_FLAC // PKP: FLAC support needs special library from VLSI PO

#define SHUFFLE_PLAY /*shuffle instead of random +12/19 words*/
#define SKIP_ID3V2    /* skip ID3v2 tags */

#ifndef NO_WMA
//#define UI_PAUSE_BEFORE_PLAY //disable pause key function, wait for play
#endif

#define USE_LCD_DISPLAY

#define TIMEOUT /* timeout when decoding seems to be stuck */
//#define ENABLE_I2S /*practical for VS1053 only*/
#define ENABLE_HIGHREF /*VS1053 only -- use if AVDD >= 3.3V */
#define HDATCHECK 800 /* 21words: give up after 800/2=400kB ~ 3-6sec */
 
//#define SAVE_POSITION
//#define SAVE_VOLUME

//#define PLAYTIME 15
//#define SAVE_VOLUME /*66 words */
//#define SAVE_POSITION /*32 words*/
#if defined(SAVE_VOLUME) || defined(SAVE_POSITION)
#define SAVE_STUFF
#endif

#include <string.h>
#include "standalone.h"
#include <stdio.h>
#include "mmcfs.h"

u_int16 GetI6(void);
auto void DecodeTimeClear(void);
void PowerOff(void);


extern __y u_int16 fileName[6];

#define FAT_LFN_SIZE 100
u_int16 longFileName[FAT_LFN_SIZE];
s_int16 fileNamePos = -20;
u_int16 fileSizeW;


typedef enum {
  spi_pause=0,
  spi_file,
  spi_seek,
  spi_waitdata,
  spi_data,
  spi_zeros,
  spi_record,
  spi_waitima
} SPISTATE;

SPISTATE spiState, oldSpiState;
u_int16 spiCnt;
#ifdef TIMEOUT
u_int16 timeOut;
#endif

struct SPIX { /* do not touch! ASM uses the structure */
  s_int32 size;
  u_int32 sector;
  u_int16 fragSize; /* TODO: make this 32-bit and multiply fragment list clusters by sectorsPerCluster. */
  __y struct FRAGMENT *fragment;
} spix;


//#define CTRL3_UPDATE_VOL        (1<<15)
#define CTRL3_BY_NAME           (1<<8) /*locate by name*/
#define CTRL3_RECORD_ON         (1<<7)
#define CTRL3_AT_END            (1<<6)
#define CTRL3_NO_NUMFILES       (1<<5)
#define CTRL3_PAUSE_ON          (1<<4)
#define CTRL3_FILE_READY        (1<<3)
#define CTRL3_PLAY_MODE_MASK    (3<<1)
#define CTRL3_PAUSE_AFTER_PLAY  (3<<1)
#define CTRL3_PAUSE_BEFORE_PLAY (2<<1)
#define CTRL3_LOOP_SONG         (1<<1)
#define CTRL3_PLAY_NEXT         (0<<1)
#define CTRL3_RANDOM_PLAY       (1<<0) /* SCI_AICTRL3 */

u_int16 screenUpdate = 1;

#define LCD_GPIO_PIN_A0 4
#define LCD_GPIO_PIN_CS 5

register __a1 short SpiDelay(void);

void Delay(u_int32 n){
  while (n--) {
    SpiDelay();
  }
}


#ifdef USE_LCD_DISPLAY

#include "lcdisplay.h"
auto u_int16 SpiSendReceiveLcd(register __a0 u_int16 dataTopAligned, register __a1 bits);


#if 0 //cheap LCD display
const unsigned char lcdInit[]={
  ST7565_BIAS_1_9,	 		/* Bias 1/9   */
  ST7565_ADC_NORMAL,		/* X direction */
  ST7565_COM_OUT_63_0,	/* Y direction */
  ST7565_V0_OUT_4,		 /* Use this to adjust the contrast */
  ST7565_ELE_VOL_SET,
  0x27,
  //ST7565_ELE_VOL_VAL(0x27), /* Use this to adjust the contrast */
  ST7565_DISPLAY_ALL_OFF,	/* Normal operation, not all on */
  ST7565_DISPLAY_NORMAL,	/* Normal display, not reverse*/
  ST7565_STATIC_INDICATOR_SET,
  ST7565_STATIC_INDICATOR_OFF,	/* Do not show indicator */
  ST7565_POWER_SET_7,		/* Set power */
  ST7565_DISPLAY_ON,		/* Display on*/
  0x00
  //ST7565_SET_START_LINE(0x00)
};
#else //OLED display
const unsigned char lcdInit[]={
  0xae, // Display off
  0xd3, // Display offset
  0x00, // 'value
  0xa8, // Multiplex ratio
  0x3f, // 'value
  0xad, // DC/DC on
  0x8b, // 'value 8b
  0x40, // Display start line
  0xa0, // Segment remap
  0xc8, // Scan direction
  0x81, // Set Contrast (=brightness)
  0x50, // 00 is minimum, ff is maximum
  0xb0, // Page address
  0x00, // Column address low nibble
  0x10, // Column address high nibble
  0xa4, // Entire display on
  0xa6, // Normal (not inverted) display
  0xaf, // Display on
  0xd8, // Power Save
  0x05  // Low power mode
};
#endif


#define SCREEN_SET_COMMAND_MODE USEX(GPIO_ODATA) &= ~(1 << LCD_GPIO_PIN_A0);
#define SCREEN_SET_DATA_MODE USEX(GPIO_ODATA) |= (1 << LCD_GPIO_PIN_A0);

#define SCREEN_END USEX(GPIO_ODATA) |= (1 << LCD_GPIO_PIN_CS);
#define SCREEN_SELECT USEX(GPIO_ODATA) &= ~(1 << LCD_GPIO_PIN_CS);


#define SYMBOL_SPACE 32
#define SYMBOL_NUMBER 48
#define SYMBOL_ALPHA 65
#define SYMBOL_ALPHA_LOW 97 
#define SYMBOL_LEFTSPK 128
#define SYMBOL_RIGHTSPK 129
#define SYMBOL_USB 130
#define SYMBOL_DISKDRIVE 131
#define SYMBOL_PU 132
#define SYMBOL_KATAKANA 133
#define SYMBOL_DIGIT 197
#define SYMBOL_NOTE 207
#define SYMBOL_PLAY 208
#define SYMBOL_PAUSE 209
#define SYMBOL_STOP 210


u_int16 screenX, screenY;
const u_int16 fontPtrs[] = {
  0x000,0x002,0x004,0x007,0x00d,0x013,0x018,0x01e,
  0x020,0x023,0x026,0x02b,0x030,0x032,0x036,0x038,
  0x03f,0x043,0x047,0x04b,0x04f,0x053,0x057,0x05b,
  0x05f,0x063,0x067,0x069,0x06b,0x06e,0x072,0x075,
  0x07a,0x080,0x085,0x08a,0x08f,0x094,0x099,0x09e,
  0x0a3,0x0a8,0x0ab,0x0b0,0x0b5,0x0ba,0x0c1,0x0c6,
  0x0cc,0x0d1,0x0d6,0x0db,0x0e0,0x0e5,0x0ea,0x0ef,
  0x0f4,0x0f9,0x0fe,0x103,0x105,0x10c,0x10e,0x113,
  0x118,0x11b,0x11f,0x123,0x127,0x12b,0x12f,0x133,
  0x137,0x13b,0x13e,0x141,0x145,0x148,0x14d,0x151,
  0x155,0x159,0x15d,0x161,0x165,0x168,0x16c,0x171,
  0x176,0x17b,0x17f,0x183,0x186,0x187,0x18a,0x18f,
  0x194,0x199,0x19e,0x1a9,0x1b4,0x1bd
};
const u_int16 fontData[] = {
  0x0000,0x0000,0x005f,0x005f,0x0007,0x7000,0x5007,0x7014,
  0x007f,0x0014,0x0014,0x007f,0x0f14,0x0104,0x012a,0x406b,
  0x406b,0x782a,0x0010,0x0032,0x1018,0x200c,0x4006,0x0013,
  0x0020,0x0056,0x1849,0x1855,0x0022,0x0050,0x0a03,0x0a01,
  0x4a1c,0x2a22,0x1e41,0x0441,0x4422,0x341c,0x142a,0x0c1c,
  0x203e,0x101c,0x782a,0x0408,0x0008,0x183e,0x0808,0x4c08,
  0x4860,0x3830,0x4808,0x4808,0x7808,0x4808,0x4860,0x4860,
  0x2840,0x1820,0x7c10,0x0808,0x0804,0x7c02,0x0801,0x283e,
  0x1841,0x4041,0x483e,0x4800,0x7842,0x407f,0x5440,0x5462,
  0x5451,0x7c49,0x0046,0x1822,0x0049,0x5849,0x4036,0x3818,
  0x1014,0x1012,0x107f,0x102f,0x1049,0x0149,0x4131,0x3d3e,
  0x0949,0x0749,0x1032,0x0801,0x7c79,0x0205,0x0103,0x0e36,
  0x0249,0x4349,0x2236,0x1e26,0x4449,0x4449,0x7c3e,0x4414,
  0x4414,0x2214,0x1234,0x0a08,0x7f14,0x0222,0x4214,0x3f14,
  0x0214,0x4214,0x3e22,0x0a14,0x0a08,0x7f02,0x0a01,0x0a59,
  0x0805,0x4602,0x423e,0x2241,0x1e49,0x0455,0x0359,0x422e,
  0x3e7c,0x020a,0x4209,0x420a,0x427c,0x427f,0x7e49,0x0249,
  0x4f49,0x2236,0x1f1c,0x0222,0x4a41,0x4a41,0x4022,0x207f,
  0x1c41,0x4241,0x2222,0x121c,0x2a7f,0x4649,0x0249,0x3f49,
  0x4241,0x4a7f,0x4609,0x0609,0x4809,0x4001,0x201c,0x1e22,
  0x0849,0x4649,0x4a3a,0x327f,0x1e08,0x0a08,0x4a08,0x3e7f,
  0x0941,0x087f,0x0641,0x0020,0x4640,0x2040,0x1e20,0x041f,
  0x457f,0x3d08,0x0514,0x0422,0x0041,0x7f7f,0x0840,0x1040,
  0x0040,0x4440,0x247f,0x1f02,0x0404,0x0408,0x4004,0x4202,
  0x427f,0x427f,0x4004,0x4208,0x2a10,0x127f,0x2a1c,0x0622,
  0x2241,0x1241,0x7b22,0x161c,0x227f,0x0009,0x4009,0x2009,
  0x1f06,0x001c,0x7822,0x0051,0x0261,0x04be,0x787f,0x3f09,
  0x4419,0x4429,0x4446,0x4426,0x0249,0x4249,0x4249,0x2232,
  0x1e01,0x0401,0x027f,0x0401,0x0801,0x303f,0x3240,0x0240,
  0x7f40,0x023f,0x320f,0x0230,0x1240,0x2230,0x520f,0x0e7f,
  0x0020,0x2a18,0x2a20,0x2a7f,0x4063,0x3814,0x2408,0x2214,
  0x2063,0x7003,0x4004,0x2878,0x1004,0x2803,0x0661,0x0a51,
  0x3e49,0x4a45,0x4a43,0x4a7f,0x0441,0x7f01,0x0402,0x1404,
  0x0c08,0x4010,0x4220,0x4240,0x7e41,0x407f,0x4a04,0x4a02,
  0x4a01,0x4a02,0x7e04,0x0480,0x0580,0x4580,0x2580,0x1c80,
  0x0f01,0x4003,0x2003,0x1f20,0x0054,0x7c54,0x0078,0x7e7f,
  0x4044,0x3044,0x7e38,0x4038,0x2044,0x1044,0x0828,0x7e38,
  0x4244,0x4244,0x427f,0x7e38,0x0e54,0x0254,0x4218,0x2200,
  0x1e88,0x427e,0x4209,0x4018,0x20a4,0x18a4,0x0278,0x047f,
  0x0104,0x0204,0x0078,0x0744,0x057d,0x0740,0x0080,0x0084,
  0x367d,0x417f,0x4110,0x4128,0x3644,0x003f,0x0040,0x0040,
  0x007c,0x3604,0x3078,0x4904,0x4978,0x497c,0x0604,0x0004,
  0x4978,0x4938,0x4944,0x3644,0x0638,0x08fc,0x0824,0x0824,
  0x3618,0x0618,0x4924,0x4924,0x49fc,0x307c,0x3604,0x4904,
  0x4908,0x4948,0x3054,0x0054,0x0124,0x0104,0x013f,0x3644,
  0x363c,0x4940,0x4940,0x493c,0x360c,0x0630,0x4940,0x4930,
  0x490c,0x363c,0x6040,0x6038,0x3f40,0x023c,0x0444,0x7f28,
  0x3e10,0x1c28,0x0844,0x001c,0x7fa0,0x7fa0,0x007c,0x7f64,
  0x7f54,0x7f4c,0x7f44,0x7f00,0x7f36,0x7f49,0x007f,0x0000,
  0x0049,0x0036,0x0008,0x0008,0x003e,0x001c,0x0008,0x0008,
  0x001c,0x003e,0x0008,0x0008,0x00ff,0x0081,0x007e,0x0024,
  0x003c,0x003c,0x0024,0x007e,0x0081,0x00ff,0x001c,0x001c,
  0x0008,0x000c,0x001a,0x002b,0x006b,0x0068,0x0008,0x001c,
  0x0008,0x00f8,0x008c,0x00aa,0x00a9,0x00a9,0x00a9,0x0089,
  0x00f9,0x0045,0x0023,0x001f,0x000c,0x004e,0x00ef,0x00e7,
  0x00d8,0x00e7,0x00ef,0x004e,0x000c
};


u_int16 screenNotification[20];
u_int16 screenNotificationTime = 0;


// Put n binary octets to display controller
void ScreenPutsN(unsigned char *c, int n){
  SCREEN_SELECT;
  while(n--) {
    SpiSendReceiveLcd((*c++)<<8,8);
  }
  SCREEN_END;
}

// Write command c to display controller
void ScreenPutCommand(unsigned char c){
  SCREEN_SET_COMMAND_MODE;
  SCREEN_SELECT;
  SpiSendReceiveLcd(c<<8,8);
  SCREEN_END;
}

// Set x,y pointer of display controller
void ScreenLocate(int x, int y) {
  screenX = x;
  screenY = y & 7;
  ScreenPutCommand((y & 7) | 0xb0);
  ScreenPutCommand(x & 0x0f);
  ScreenPutCommand(((x >> 4) & 15) | 0x0010);
}


// Write image data octet to display controller
void ScreenPutData(u_int16 c){
  SCREEN_SET_DATA_MODE;
  SCREEN_SELECT;
  SpiSendReceiveLcd((c<<8),8);
  SCREEN_END;
  screenX++;
}

// Write double height image data octet to display controller 
void ScreenPutDoubleData(u_int16 c){
  register u_int16 d;
  register u_int16 i;

  d=0;
  for (i=0; i<8; i++){
    d <<= 1;
    d |= ((c & 0x80) >> 7);
    d <<= 1;
    d |= ((c & 0x80) >> 7);
    c <<= 1;     
  }

  ScreenPutData(d);
  ScreenPutData(d);
  ScreenLocate(screenX-2, screenY+1);
  ScreenPutData(d>>8);
  ScreenPutData(d>>8);  
  ScreenLocate(screenX, screenY-1);
  
}

#define ScreenPutChar(c) ScreenOutputChar(c,ScreenPutData)
#define ScreenPutBigChar(c) ScreenOutputChar(c,ScreenPutDoubleData)

// Draw character c to screen using custom data output function
// This function is responsible for character mapping.
void ScreenOutputChar(register u_int16 c, void (*OutputFunction)(u_int16 c) ){
  u_int16 length;
  register u_int16 i;
  register u_int16 bit_index;
  register const u_int16 *p;

  if ((c<32)) { //Invalid char
    return;
  }    

  if ((c>210)) { //Invalid char
    return;
  }    

  if (c>132){ //Fixed width charset
    length = 5;    
    bit_index = 8;
    p = fontData+((c-133)*5);
  } else { //variable width charset ("normal" ascii)
    c -= 32; // ASCII ' ' (32 decimal) is in ROM index 0
    length = fontPtrs[c+1] - fontPtrs[c];
    bit_index = 0;
    p = &fontData[fontPtrs[c]];
  }

  if (screenX>131){
    screenY++;
    ScreenLocate(0,screenY);
  }

  for(i=0;i<length;i++){
    OutputFunction((*p++) >> bit_index);
  }
  OutputFunction(0);
}



// Screen Clear
void ScreenClear(){
  u_int16 x,y;
  for (y=0; y<8; y++){
    ScreenLocate(2,y);
    for (x=0; x<132; x++){
      ScreenPutData(0);
    }
  }    
  ScreenLocate(0,0);
}

// Put packed string in Y ram (mainly for file name output)
void ScreenPutPacked(const char *d, u_int16 xEnd){
  int i;

  for (i=0;;i++){      
    register u_int16 c;
    c=*d++;
    if ((c & 0xff00U) == 0){
      break;
    }
    ScreenPutChar(c>>8);
    if ((c & 0xffU) == 0){
      break;
    }
    ScreenPutChar(c&0xff);
  }  
    
  while (screenX <= xEnd){
    ScreenPutData(0);
  }            
}


u_int16 valueString[8];
auto void IntToString(u_int16 value) {
  register u_int16 i; 
  for (i=0; i<8; i++){
    valueString[i]=value % 10;
    value = value / 10;
  }
}
 
// Draw a u_int16 value to the screen as decimal number
void ScreenPutUInt(register u_int16 value, register u_int16 length){
  IntToString(value);
  do {
    ScreenPutChar(SYMBOL_NUMBER + valueString[--length]);
  } while (length);     
}


// Draw a u_int16 value to the screen as decimal number in large numbers
void ScreenPutBigInt(register u_int16 value, register u_int16 length){
  register u_int16 i; 
  u_int16 s[8];

  for (i=0; i<8; i++){
    s[i]=value % 10;
    value = value / 10;
  }
  do {
    ScreenPutBigChar(SYMBOL_DIGIT + s[--length]);
  } while (length);     
}


// Draw a volume bar (triangle) to screen
void VolumeBar(u_int16 volume){
  register u_int16 i;
  for (i=0; i<32; i++){
    register u_int16 d=0xff80;
    if (volume>i) {
      d >>= (i>>2)+1;
    }
    d &= 0x7f;
    ScreenPutData(d);
  }
}


// Draw a progress bar to the screen

// Allow code optimization by making the progress bar always start at
// the leftmost edge of screen
#define xMin 0

void ScreenPutBar(register u_int32 currentValue, u_int32 maxValue, u_int16 xMax){
  //u_int16 xMin=screenX;
  u_int16 xUntil=0;
  if (maxValue==0){
    maxValue=1;
  }
  if (currentValue>maxValue){
    currentValue=maxValue;
  }  
  xUntil=xMin+((currentValue*(xMax-xMin-4))/maxValue);

  
  ScreenPutData(12); 
  ScreenPutData(30);
  ScreenPutData(63); 
  ScreenPutData(127);
  
  while (screenX < (xMax-4)){
    register u_int16 d=161; //frame pixels
    if (screenX<xUntil){
      d |= 30; //fill pixels
    }
    ScreenPutData(d);
  }
    
  ScreenPutData(146);
  ScreenPutData(140);
  ScreenPutData(72);
  ScreenPutData(48);
}

// Write zero data (black pixels) to display
auto void ScreenZeros (u_int16 n){
  while (screenX < n) {
    ScreenPutData(0);
  }
}


// Start the display controller
void ScreenInit(){
  USEX(GPIO_DDR) |= (1 << LCD_GPIO_PIN_A0) | (1 << LCD_GPIO_PIN_CS) | (1 << 3); //3=SD_CLK
  SCREEN_SET_COMMAND_MODE;
  ScreenPutsN(lcdInit,sizeof(lcdInit));
  SCREEN_END;  

  ScreenClear();  

  ScreenLocate(20,2);
  ScreenPutPacked("\pVLSI Solution VS1053",130);
}


u_int16 screenPhase = 0;


void ScreenPutVolts() {
  static u_int32 voltMeas;
  static u_int16 lowVoltage = 0;  
  static u_int16 myPhase = 0;
  static u_int16 voltMeasCount = 0;
  
  
  myPhase++;

  USEX(DECIM_CONTROL) = DECIM_ENABLE | DECIM_FACTOR48K; // Decimator on, factor, enabled.

  if (16 == (myPhase & 0x1f)) {
    if (lowVoltage) {
      ScreenLocate(101,0);
      ScreenZeros(131);
    }
  }    


  voltMeas += USEX(DECIM_DATA_RIGHT);
  voltMeasCount++;
  
  if (0 == (myPhase & 0x1f)){  
    // Battery voltage

    if (voltMeasCount) {
      // calculate average of voltMeas
      voltMeas /= voltMeasCount;
    }

    // Linear convert into millivolts
    voltMeas += 8122;
    voltMeas *= 1000;
    voltMeas /= 2500;

    ScreenLocate(101,0);
    ScreenPutChar(voltMeas/1000+'0');
    ScreenPutChar('.');
    ScreenPutUInt(voltMeas%1000,3);
    ScreenPutData(0);
    ScreenPutChar('V');

    // Is battery voltage below 3.7 volts? Then set lowVoltage flag.
    lowVoltage = (voltMeas < 3700) ? 1 : 0;

    // Is battery voltage below 3.6 volts? Then set power off.
    if (voltMeas < 3600) {
      static u_int16 powerOffCount = 0;
      if (++powerOffCount > 10) {
	PowerOff();
      }
    }

    voltMeas = 0;    
    voltMeasCount = 0;
  }    
}


void ScreenUpdate() {
  
  // Song Number of Shuffle
  ScreenLocate(0,0);
  if (USEX(SCI_AICTRL3) & CTRL3_RANDOM_PLAY) {
    ScreenPutPacked("\pShuffle",37);    
  } else {
    ScreenPutUInt(USEX(SCI_AICTRL0)+1,4);
  }
  ScreenZeros(37);

  // Decode Time
  ScreenLocate(38,4);
  ScreenPutBigInt(USEX(SCI_DECODE_TIME)/60, 2);
  ScreenPutBigChar(':');
  ScreenPutBigInt(USEX(SCI_DECODE_TIME)%60, 2);
  

  ScreenPutVolts();
  screenPhase++;
  
  ScreenLocate(0,2);

  if (screenPhase & 1)

    if (screenNotificationTime) {
      screenNotificationTime--;
      ScreenPutPacked(screenNotification,131);
    } else {
      {
	u_int16 i,j = 0;
	
	fileNamePos++;
	if (fileNamePos > -20){
	  j = (fileNamePos<0) ? 0 : fileNamePos;
	  while (longFileName[j]){
	    if (screenX > 126) {
	      goto nexit;
	    }
	    ScreenPutChar(longFileName[j++]);
	  }
	  fileNamePos = -40;
	nexit:
	  ScreenZeros(131);
	}
      }
    }
  

#if 0
  // Show Sample Rate
  ScreenLocate(70,0);
  ScreenPutUInt(hwSampleRate/1000,2);
  ScreenPutData(0);
  ScreenPutPacked("\pkHz",3);
#endif
	
  if (1 == (screenPhase & 0x1f)){
    
    // File name extension
    ScreenLocate(40,0);
    ScreenPutChar(fileName[4] >> 8);
    ScreenPutChar(fileName[4] & 0xff);
    ScreenPutChar(fileName[5] >> 8);
    ScreenZeros(69);
    
    // Pos
    ScreenLocate(0,7);
    ScreenPutBar(fatInfo.fileSize-spix.size*2, fatInfo.fileSize,130);      
    
    // Bitrate
    ScreenLocate(70,0);
    ScreenPutUInt(parametric_x.byteRate/125,3);
    ScreenPutData(0);
    ScreenPutChar('k');
    ScreenZeros(100);
  }        
  
}


#else
#define ScreenInit();
#define ScreenLocate(a,b);
#define ScreenPutPacked(a,b);
#define ScreenPutChar(a);
#define ScreenPutBigChar(a);
#define ScreenPutVolts();
#define ScreenClear();
#define ScreenPutCommand(a);

#endif


void PowerOff(void) {
  SpiSendClocks();
  USEX(SCI_STATUS) |= (1<<SCIST_APDOWN2); /*drivers off*/

  // Display off
  ScreenClear();
  ScreenLocate (40,4);
  ScreenPutPacked("\pPower Off.",128);
  Delay(2000000);
  ScreenPutCommand(0xae); //Display Off
  Delay(1000000);
  ScreenPutCommand(0xad); //DC-DC off
  ScreenPutCommand(0x8a); //value

  USEX(SER_DREQ) = 0; /* Led / Aux key scan off*/

  // Wait until power button not pressed    
  do {
    USEX(WDOG_RESET) = WDOG_RESET_VAL;
  } while ((USEX(GPIO_IDATA) & (1 << 10)) != 0);
    

  USEX(SCI_STATUS) |= (1<<SCIST_APDOWN1)|(1<<SCIST_APDOWN2);
  while (1)
    ;
}





extern volatile s_int16 *srcWp, *srcRp;
extern __y u_int16 g_dcthi[2048];
void putch(register __a0 int ch);
auto void play_frame(void);
//void Disable();
//void Enable();
//void Reset();

#include <stdlib.h>

//#undef ENABLE_WATCHDOG

#if 0 && defined(VS1003) && defined(RECORDER)
#define LOWER_LOOPBACK_VOL 0x0000
#endif

//#define USE_DEBUG

#if defined(USE_DEBUG) && !defined(UART_BUFFERED)
#define ENABLE_SCI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_DAC))
#define ENABLE_SCI_SDI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_SDI)|(1<<INT_EN_DAC))
#else
#define ENABLE_SCI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_DAC)|(1<<INT_EN_RX))
#define ENABLE_SCI_SDI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_SDI)|(1<<INT_EN_DAC)|(1<<INT_EN_RX))
#endif


#if 0
#define WAIT_FOR_BUTTON /*After poweron or watchdog reset*/
#define LOUDNESS_ALWAYS_ON_OFF
#endif


#ifndef RECORDER
#define FIX_WAV_DECODETIME  /* SCI_UI: advance decodetime for short WAV's */
#endif

//#define HDATCHECK 800 /* 21words: give up after 800/2=400kB ~ 3-6sec */


#ifndef NOCHECKFILETYPE
#define CHECKFILETYPE /* vs1011 29 words? check file type according to short-filename suffix */
#endif/*NOCHECKFILETYPE*/
//#define KEEP_DRIVERS_OFF /* .. until MMC located */
#define WITH_SUBDIRECTORIES /* support subdirectories */


#ifdef USE_HC
s_int16 hcShift;
#endif


#ifdef CHECKFILETYPE
#undef CHECKFILETYPE /*temporary undef*/
#define CHECKFILETYPE CheckFileType1053
#endif

#ifndef GPIO_IDATA
#define GPIO_IDATA GPIO_DATA
#endif

#define SPI_xCS   1 /* GPIO */
#define SPI_CLK   8
#define SPI_MISO  4
#define SPI_MISO_SHIFT -3
#define SPI_xCS2  2
#define SPI_MOSI  1 /* DREQ */
#define SPI_READ  0x03

void puthex(u_int16 d);
#ifdef USE_DEBUG
static char hex[] = "0123456789ABCDEF";
void puthex(u_int16 d) {
  register i;
  for (i=0;i<4;i++) {
    putchar(hex[(d>>12)&15]);
    d <<= 4;
  }
}
void puthex2(u_int16 d) {
  register i;
  putchar(hex[(d>>4)&15]);
  putchar(hex[(d>>0)&15]);
}
#endif

#ifdef FATINFO_IN_Y
__y struct FATINFO fatInfo;
#else
struct FATINFO fatInfo;
#endif


//#define MAX_FRAGMENTS 28//16//22
__y struct FRAGMENT fragments[MAX_FRAGMENTS];

u_int16 gFileNum[2];
extern __y s_int16 mallocAreaY[9216];


/*147 words*/
auto s_int16 HandleDir(register __i2 __y struct FRAGMENT *curFragment,
		       __y struct FRAGMENT *nextFragment) {
  register __c1 int i = 0;
  register u_int32 currentSector = curFragment->start /*& 0x7fffffffUL */;
  static u_int16 lfnReset = 0;
  
#ifdef ENABLE_WATCHDOG
  USEX(WDOG_RESET) = WDOG_RESET_VAL;
#endif
  goto first;
  
  while (1) {
    register __d1 u_int16 fn = GetByte(i+0); /* first char of filename */
    
    /* We are now looking at FAT directory structure. */
    /* Is current file a regular file? */
    if (fn == 0)
      goto eod; /*end of directory*/
    if (fn != 0xe5) { /* Not deleted */
      
      
      
      // Long file name
      if ((GetByte(i+11/*Attr*/) & 0x3f) == 0x0f) {
	register u_int16 idx = ((fn & 0x3f) - 1) * 13;
	static const short places[] = {
	  1, 3, 5, 7, 9, 14, 16, 18, 20, 22, 24, 28, 30, -1
	};
	register const short *p = &places[0];
	while (*p >= 0 && idx < FAT_LFN_SIZE-1 /* space for NUL */) {
	  longFileName[idx++] =
	    GetByte(i + *p++);
	}
	/* last long entry ? */
	if ( fn & 0x40 /*LAST_LONG_ENTRY*/ ) {
	  if (idx >= FAT_LFN_SIZE)
	    idx = FAT_LFN_SIZE-1;
	  longFileName[idx] = '\0';
	}
	lfnReset = 0;
      } else {
	if (++lfnReset==2) {
	  longFileName[0] = 0;
	}
      }
      
      
#ifdef WITH_SUBDIRECTORIES
      /* Is it a subdirectory? */
      if ((GetByte(i+11/*Attr*/) & 0x10) && fn != '.') {
	/* TODO: do not call if FAT12! */
	if (HandleDir(nextFragment,
		      FragmentList(nextFragment,
				   ((u_int32)GetWord(i+20)<<16)+GetWord(i+26))) < 0)
	  return -1;
	ReadDiskSector(currentSector);/*reread sector*/
      } else /* it was a subdirectory */
#endif
	/* Attributes: NO directory, NO volume id, NO system, NO hidden */
	if ((GetByte(i+11/*Attr*/) & 0xde) == 0
#ifdef CHECKFILETYPE
	    /*
	      check supported files from name suffix:
	      vs1011/vs1002:  wav mp3
	      vs1003:         wav mp3 mid wma asf
	      vs1033: wav mp3 mid wma asf mp4 aac m4a
	    */
	    && CHECKFILETYPE(GetLong(i+8))
#endif
	    ) {
	  
	  /* It is a regular file. */
	  
#if defined(SCI_UI)
	  /* If open-by-name, check name and open file if match. */
	  if (USEX(SCI_AICTRL3) & CTRL3_BY_NAME) {
	    register int j, t = 0;
	    register u_int16 *p = stream_buffer + i/2;
	    register __y u_int16 *d = fileName;
	    p[5] &= 0xff00; /* clear the attribute byte */
	    for (j=0; j<6; j++) {
	      t |= *d++ ^ *p++;
	    }
	    if (t == 0) {
	      USEX(SCI_AICTRL0) = gFileNum[1];
	      goto thisfile;
	    }
	  }
#endif
	  
	  gFileNum[1]++; /* total file count */
	  if (gFileNum[0]-- == 0) {
	  thisfile:
	    //#ifdef SCI_UI
	    /* Put the 8.3-character filename to memory so the user
	       can read it. Two bytes per word, big-endian. */
	    {   /* 13 words */
	      register int j;
	      register u_int16 *p = stream_buffer + i/2;
	      register __y u_int16 *d = fileName;
	      for (j=0;j<6;j++) {
		*d++ = *p++;
	      }
	    }
	    //#endif
	    /* ------------ FILE FOUND ------------- */
	    fatInfo.fileSize = GetLong(i+28); /*do first!*/
	    FragmentList(&fragments[0],
			 ((u_int32)GetWord(i+20)<<16) + GetWord(i+26));
	    return -1; /* File found, All OK return */
	  }
	} /* normal file */
    } /* 0xe5 / 0x00 deleted or unused entry */
    i = (i + 32) & 511;
    if (i == 0) {
      currentSector++;
      if (--(curFragment->size) == 0) {
	if ((s_int16)(curFragment->start>>16) < 0) {
        eod:
#if defined(SCI_UI)
	  USEX(SCI_AICTRL3) &= ~CTRL3_BY_NAME;
#endif
	  return gFileNum[1];
	}
	curFragment++;
	currentSector = curFragment->start /*& 0x7fffffffUL*/;
      }
    first:
#if 0 && defined(USE_DEBUG)
      puthex(currentSector);
      puts(" read");
#endif
      ReadDiskSector(currentSector);
    }
  }
}


#ifdef HDATCHECK
u_int16 hdatCheck;
#endif
extern __y u_int16 hwSampleRate;

/* Can be done for all formats, because IdleHook is run as the main thread
   and not in interrupts. Whenever IdleHook is called, there either is
   not enough stream data anyway, or the decoder is outputting samples
   and there is not enough space in the audio buffer (thus not reading
   stream). */
u_int16 __x * __x stream_save;

#ifndef SCI_UI
__y struct {
  u_int16 Data; /* must be first entry! Used by c.s */
  u_int16 Cnt;
  u_int16 Strobe;
  u_int16 Timer; /* Data,Cnt,Strobe,Timer must be in this order (buts.s) */
  u_int16 Key;
  s_int16 KeyTime;
} ui;
#endif

#ifdef ENABLE_UI
s_int16 ApplAddr(register __i0 s_int16 **d, register __a1 s_int16 mode,
		 register __a0 s_int16 n);
s_int16 ApplAddrButRef(register __i0 s_int16 **d, register __a1 s_int16 mode,
		       register __a0 s_int16 n);
#endif/*ENABLE_UI*/

s_int16 ApplAddrSci(register __i0 s_int16 **d, register __a1 s_int16 mode,
		    register __a0 s_int16 n);
s_int16 ApplAddrRef(register __i0 s_int16 **d, register __a1 s_int16 mode,
		    register __a0 s_int16 n);



#if defined(FIX_WAV_DECODETIME) && defined(SCI_UI)
u_int16 lastDecodeTime;
#endif /*FIX_WAV_DECODETIME + SCI_UI*/

void SaveWord(register __c0 u_int16 addr, register __c1 u_int16 value);
#ifdef SAVE_VOLUME
int volChanged = 0;
#endif

#ifdef SAVE_STUFF
#define SPI_WREN  0x06
#define SPI_WRDI  0x04
#define SPI_RDSR  0x05
#define SPI_WRSR  0x01
#define SPI_READ  0x03
#define SPI_WRITE 0x02
  
auto u_int16 SpiSendReceiveSpi(register __a0 u_int16 datTopAligned, register __a1 s_int16 bits);
  
#if 1
void SaveWord(register __c0 u_int16 addr, register __c1 u_int16 value);
#else
void SaveWord(register __c0 u_int16 addr, register __c1 u_int16 value) {
  USEX(GPIO_ODATA) &= ~(SPI_xCS2|SPI_xCS);
  SpiSendReceiveSpi(SPI_WREN<<8, 8);
  USEX(GPIO_ODATA) |= SPI_xCS;
  USEX(GPIO_ODATA) &= ~SPI_xCS;
  SpiSendReceiveSpi(SPI_WRITE<<8, 8);
  SpiSendReceiveSpi(addr/*volume=26*/, 16);
  SpiSendReceiveSpi(value, 16);
  USEX(GPIO_ODATA) |= SPI_xCS|SPI_xCS2;//SPI_CLK;
}
#endif
//SaveWord(26/*volume=26*/, USEX(SCI_VOL));
#endif


#ifdef SHUFFLE_PLAY
u_int16 shuffle_offset = 0;
#endif

#define USE_REWIND
#ifdef USE_REWIND
int rewmode = 0;
int rewcount = 0;
#endif



/*
  Called before HALT
*/
void UserHook(void) {
  register __c1 SPISTATE cachedSpiState = spiState;
  
  static u_int16 holdOn = 0;
  static u_int16 modeFlashTimer = 0;
  

#ifndef SCI_UI      
  if (ui.Strobe) {
    // Real-Time Tick
  }  
#endif
    
#define SW1 1
#define SW2 2
#define SW3 3
#define SW4 4
#define SW5 5
#define SW6 6
#define SW7 7
#define SW8 8
    
#define SW_HOME SW1
#define SW_PREV SW2
#define SW_STOP SW3
#define SW_VOL_UP SW4
#define SW_VOL_DN SW5
#define SW_REC SW6
#define SW_NEXT SW7
#define SW_PLAY SW8
#define SW_OK SW_PLAY
#define SW_NO SW_STOP
#define SW_UP SW_VOL_MORE
#define SW_DOWN SW_VOL_LESS
#define SW_LEFT SW_PREV
#define SW_RIGHT SW_NEXT
#define SW_HOLD SW_STOP
#define SW_MODE SW_STOP

  //#define SW4 0x4000U
      
#ifdef ENABLE_UI
        
  // PKP: -- UI: USER INTERFACE --
          
  if (ui.Strobe) { // PKP: It's time to check the buttons and do UI calculations
    
    screenUpdate = 1;
    ui.Data = ui.Strobe = 0;
    // PKP: -- UI: READ KEYS -- 
    
    USEX(SER_DREQ) = 0;		
    ui.Data = 
      USEX(GPIO_IDATA) & (1 << 10) ? SW1 :
      USEX(GPIO_IDATA) & (1 <<  8) ? SW2 :
      USEX(GPIO_IDATA) & (1 << 11) ? SW3 :
      USEX(GPIO_IDATA) & (1 <<  6) ? SW4 : 0;
    
    
    if (!ui.Data) {
      USEX(SER_DREQ) = 1;
      ui.Data = 
	USEX(GPIO_IDATA) & (1 << 10) ? SW5 :
	USEX(GPIO_IDATA) & (1 <<  8) ? SW6 :
	USEX(GPIO_IDATA) & (1 << 11) ? SW7 :
	USEX(GPIO_IDATA) & (1 <<  6) ? SW8 : 0;		  
    }
    
    
    
    
    // PKP: -- UI: KEY FUNCTIONS --
    // User Interface functions
    
    parametric_x.playSpeed = 1;
    
    if (holdOn) {
      if (ui.Data != SW_HOLD) { 
	ui.Data = 0; //In HOLD mode, block everything except hold
      }
    }
    
    if (ui.Data == ui.Key) { // PKP: Current Keystate is same as previous Keystate
      if (ui.KeyTime < 8) {
	ui.KeyTime++;
      } else // PKP: keystate has been consistent for 8 ticks - it's a long keypress //
	if (ui.Key
          #ifdef WAIT_FOR_BUTTON
	    && ui.KeyTime > 0
          #endif
	    ) {

	  /* long presses */
	  register u_int16 vol = USEX(SCI_VOL);
	  switch(ui.Key) {
	    

	  case SW_HOME: //Long press of HOME
	    PowerOff();
	    break;
	    
	  case SW_VOL_UP: //Long press of Volume Up
	    if (vol) {
	      vol -= 0x0101;
	      goto lsetvol;
	    }
	    break;
	  
	  case SW_VOL_DN: //Long press of Volume Down
	    if (vol != (s_int16)0xffffU)
	      vol += 0x0101;
	  lsetvol:

	    USEX(SCI_VOL) = vol; /*in VS1011E disables ints! */
            #ifdef SAVE_VOLUME
	      volChanged = 1;
            #endif/*SAVE_VOLUME*/
	    break;
	    
	  
	  case SW_HOLD: //Long press of Hold
	    if (ui.KeyTime == 8) {
	      ui.KeyTime = 9;
	      // toggle HOLD
	      #ifdef USE_LCD_DISPLAY
	        screenNotificationTime = 20;
		if (holdOn) {
		  holdOn = 0;
		  strcpy(screenNotification,"\pKeylock Off.");
		  
		} else {
		  holdOn = 1;
		  strcpy(screenNotification,"\pKeylock On.");
		}
	     #else
		if (holdOn) {
		  holdOn = 0;
		} else {
		  holdOn = 1;
		}
	     #endif
	    }	  

	    break;	    
	    
#ifdef USE_REWIND
	    
	  case SW_PREV:
	    if (rewmode == 0) {
	      rewmode = 1;
	    }
	    break;
	      	    
#endif
	  
	  case SW_NEXT:  // Long press of NEXT -> FF
	    parametric_x.playSpeed = 6;
	    break;

	  default:
	  } //end switch-case	  	 	  
	}
      
    } else { // PKP: Keystate is not the same as previous keystate
      // PKP: Handle short keypresses

      if (ui.Data == 0
        #ifdef WAIT_FOR_BUTTON
	  && ui.KeyTime > 0
        #endif
	  ) {
	// PKP: Handle end of short keypress (current keystate ui.Data=0)
	/* short presses */
	if (ui.KeyTime < 8) {
	  /* was pressed < 0.5 seconds */
	  register u_int16 vol = USEX(SCI_VOL);

	  switch (ui.Key) {
	  
	  case SW_VOL_UP:
	    if (vol>0x0403) {
	      vol -= 0x0404;
	      goto lsetvols;
	    }
	    break;
	  
	  case SW_VOL_DN:
	    if (vol != (s_int16)0xffffU)
	      vol += 0x0404;
	  lsetvols:
	    USEX(SCI_VOL) = vol; /*in VS1011E disables ints! */
#ifdef SAVE_VOLUME
	    volChanged = 1;
#endif/*SAVE_VOLUME*/
	    break;
	  
	  
	  case SW_MODE: //Short press of Mode
	    modeFlashTimer = 16;
	    
	    // Toggle shuffle play
	    USEX(SCI_AICTRL3) ^= CTRL3_RANDOM_PLAY;
	    if (cachedSpiState == spi_pause) {
	      cachedSpiState = oldSpiState;
	      stream_wr_pointer = stream_save;
	    }		
	    break;
	  
	    
	  


	  case SW_NEXT: //intentional fall-through
	  case SW_PREV:
	    {
	      register u_int16 song = USEX(SCI_AICTRL0);
	      if (ui.Key == SW_NEXT) {
		song++; /* next file */
	      } else
               #ifndef UI_PAUSE_BEFORE_PLAY
		if (USEX(SCI_DECODE_TIME) < 5) //Song played for less than 5 seconds
               #endif
		  {
		    if (song == 0) {
		      song = USEX(SCI_AICTRL1)-1;
		    } else { /* previous file */
		      song--;
		    }
		    /* else the same file */
		}
	      USEX(SCI_AICTRL0) = song;
	      
	      /* especially needed for MIDI!! */
              #if !defined(NO_MIDI) && (!defined(ENABLE_LAYER12) || !defined(HDATCHECK))
	        USEX(SCI_MODE) |= (1<<SCIMB_OUT_OF_WAV);
              #endif
	      cachedSpiState = spi_zeros;
              #if defined(ENABLE_LAYER12)
	        /* gain=0 for partial last frame (bugs) */
	        volume[0] = volume[1] = 0;
              #endif		       			
	    }
	    break;
	    
	  
	  case SW_PLAY:
	    /* play pause/continue -- not very good for midi.. */
	    if (cachedSpiState != spi_pause) {
              #ifndef UI_PAUSE_BEFORE_PLAY
	        oldSpiState = cachedSpiState;
	        cachedSpiState = spi_pause;
	        stream_save = stream_wr_pointer;
	        stream_wr_pointer = stream_rd_pointer+1;
              #endif/*!UI_PAUSE_BEFORE_PLAY*/
	    } else {
	      cachedSpiState = oldSpiState;
	      /* restore 'stolen' data */
	      stream_wr_pointer = stream_save;
	    }
	    break;
	  default:
	  } //end of switch-case

	}//(ui.keyTime was < 8) (Short keypress)

      }
      ui.Key = ui.Data;
      ui.KeyTime = 0;
    }
    
    
#if defined(SAVE_VOLUME)
    /*Save volume when volume button is released. */
    if (ui.Data == 0 &&
	volChanged &&
	cachedSpiState == spi_seek) {
      volChanged = 0;
      SaveWord(26/*volume=26*/, USEX(SCI_VOL));
    }
#endif/*SAVE_VOLUME*/
    
  }
#endif /*USE_UI*/

  
#ifdef PLAYTIME
  if (USEX(SCI_DECODE_TIME) > PLAYTIME && cachedSpiState != spi_zeros) {
    USEX(SCI_AICTRL0)++;
#if !defined(NO_MIDI) && (!defined(ENABLE_LAYER12) || !defined(HDATCHECK))
    USEX(SCI_MODE) |= (1<<SCIMB_OUT_OF_WAV);
#endif
    cachedSpiState = spi_zeros;
  }
#endif/*PLAYTIME*/
  

  
  if (cachedSpiState == spi_file) {
    
    if (0==longFileName[0]) {
      u_int16 i;
      u_int16 *p = &longFileName[0];
      for (i=0; i<4; i++){
	*p++ = fileName[i] >> 8;
	*p++ = fileName[i] & 0xff;
      }
      *p=0;
    }      


    /* extra byte read if odd file size */
    spix.size = (fatInfo.fileSize+1)/2; /* in words! */
    //	fileSizeK = spix.size/512;
    cachedSpiState = spi_seek;
  setfrag:
    spix.fragment = &fragments[0];
    spix.sector   = fragments[0].start /*& 0x7fffffffUL*/;
    spix.fragSize = fragments[0].size; /* *fatInfo.sectorsPerCluster*/
  }
#if defined(FIX_WAV_DECODETIME) && defined(SCI_UI)
  if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) == CTRL3_LOOP_SONG &&
      USEX(SCI_HDAT1) == 0x7665/*(('v'<<8)|'e')*/) {
    if (USEX(SCI_DECODE_TIME) < lastDecodeTime)
      USEX(SCI_DECODE_TIME) = lastDecodeTime;
  }
#endif /*SCI_UI and FIX_WAV_DECODETIME*/
  
  
  if (cachedSpiState == spi_seek) {
    // MMC/SD card sector read is over, SPI bus can be used for other operations
    
    
    SpiSendClocks(); /*Raise MMC/SD CS (and send some clocks) */
    
    USEX(GPIO_ODATA) |= SPI_xCS2;
    
#ifdef USE_LCD_DISPLAY
    // Check that the audio buffer is ~full before updating screen
    if (screenUpdate & (AudioBufFill()>4000)) {
      ScreenUpdate();
      screenUpdate = 0;
    }
#endif     
    SpiSendClocks();
    
    
#define REWIND_STEP_SIZE 128
#ifdef USE_REWIND
    if (rewmode == 1) {
      s_int32 skip = spix.fragment->size - spix.fragSize;
      if (skip > REWIND_STEP_SIZE)
	skip = REWIND_STEP_SIZE;
      spix.sector -= skip;
      spix.fragSize += skip;
      spix.size += skip*512/2;
      
      rewcount = REWIND_STEP_SIZE / 8;
      rewmode = 2;
    } else if (rewmode == 2) {
      if (--rewcount == 0)
	rewmode = 0;
    }
#endif
    
#ifndef SCI_UI
    /* In case we are just trying to find something to play,
       also read keys once in a while.. */
    ui.Cnt += 64;
    if (ui.Cnt > hwSampleRate/16) {
      ui.Cnt -= hwSampleRate/16;
      ui.Timer++;
      ui.Strobe = 1;
    }
#endif
    
#ifdef USE_HC
    MmcCommand(MMC_READ_SINGLE_BLOCK|0x40,
	       (spix.sector & 0x7fffffffUL) << hcShift);
#else
    MmcCommand(MMC_READ_SINGLE_BLOCK|0x40,(spix.sector<<9));
#endif
    cachedSpiState = spi_waitdata;
    spiCnt = 0;
    NextSector();
#ifdef HDATCHECK
    if (++hdatCheck == HDATCHECK && USEX(SCI_HDAT1) == 0) {
      /* if file not detected at HDATCHECK/2 kB, skip the rest */
      goto end_of_file;
    }
#endif
  }
  
#ifndef VOICE_ACTIVATED
  /*TODO: handle spi_waitdata and spi_data
    in timer interrupt to increase throughput for FLAC */
  if (cachedSpiState == spi_waitdata) {
    if (SpiSendReceiveMmc(0xff00,8) == 0xfe) {
      cachedSpiState = spi_data;
      spiCnt = 0;
#ifdef TIMEOUT
      timeOut = 0;
#endif
    } else {
      if ((s_int16)++spiCnt < 0) {
	goto goreset; //MyReset(); /* timeout */
      }
    }
  } else if (cachedSpiState == spi_data) {
    while (spix.size > 0) {
      register int i, j = 32;
      register u_int16 *wr;
#if !defined(RECORDER) && defined(USE_FLAC)
      if (USEX(SCI_HDAT1) == (('f'<<8) | 'L')) {
	register s_int16 t = stream_wr_pointer - stream_rd_pointer;
	if (t < 0)
	  t += 0x1800;
	if (t > 0x1800/*STREAM_BUFFER_SZ*/-40)
	  break;
	wr = (void *)stream_wr_pointer;
	if (j > spix.size)
	  j = spix.size;
	for (i=0;i<j;i++) {
	  *wr++ = SpiSendReceiveMmc(-1/*0xffffU*/,16);
	  if (wr >= &stream_buffer[0x1800/*STREAM_BUFFER_SZ*/])
	    wr = stream_buffer;
	}
      } else
#endif
	{
	  if (StreamDiff() >= STREAM_BUFFER_SZ-40)
	    break;
	  wr = (void *)stream_wr_pointer;
	  if (j > spix.size)
	    j = spix.size;
	  for (i=0;i<j;i++) {
	    *wr++ = SpiSendReceiveMmc(-1/*0xffffU*/,16);
	    if (wr >= &stream_buffer[STREAM_BUFFER_SZ])
	      wr = stream_buffer;
	  }
	}
      stream_wr_pointer = (void *)wr;
      spix.size -= j;//32;
      spiCnt += 32;
      if (spiCnt >= 256) {
	/* 256 words read, get next block in file */
	cachedSpiState = spi_seek;
	SpiSendReceiveMmc(-1/*0xffff*/,16); /* discard crc */
	break;
      }
#ifdef TIMEOUT
      timeOut = 0;
#endif
      /* watchdog clearing here doesn't work with midi */
    }
    
#ifdef TIMEOUT
    if ((s_int16)++timeOut == -1) {
      spix.size = 0;
    }
#endif
    
    if (spix.size < 0x20000L) {
      /* Workaround for WMA resync after stream ended! */
      parametric_x.resync = 0;
    }
    
    
#ifdef ENABLE_WATCHDOG
    USEX(WDOG_RESET) = WDOG_RESET_VAL;
#endif
    if (spix.size <= 0) {
    end_of_file:
      SpiSendClocks();
      
#ifdef SCI_UI
      if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) == CTRL3_LOOP_SONG){
	/* restart the same file immediately */
#if defined(FIX_WAV_DECODETIME) && defined(SCI_UI)
	lastDecodeTime = USEX(SCI_DECODE_TIME)+1;
#endif /*SCI_UI and FIX_WAV_DECODETIME*/
	cachedSpiState = spi_file;
#ifdef UART_UI
	putch('l');//USEX(UART_DATA) = 'l';
#endif
      } else
#endif/*SCI_UI*/
	{
#if 0
	  USEX(SCI_AICTRL0)++; /* next file */
	  /* Just zeros, no outofwav */
	  cachedSpiState = spi_zeros;
#else
#ifdef SCI_UI
	  /*NEW: pause after play */
	  if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) ==
	      CTRL3_PAUSE_AFTER_PLAY) {
	    USEX(SCI_AICTRL3) |= CTRL3_PAUSE_ON | CTRL3_AT_END;
	  }
#else
	  {
	    register int m = (USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK);
	    if (m == CTRL3_LOOP_SONG) {
	      /* restart the same file immediately */
	      cachedSpiState = spi_file;
	    } else if (m == CTRL3_PAUSE_AFTER_PLAY) {
	      oldSpiState = spi_zeros;
	      cachedSpiState = spi_pause;
	    } else {
	      USEX(SCI_AICTRL0) += 1; /* next file */
	      cachedSpiState = spi_zeros;
#if 1
	      if (USEX(SCI_HDAT1) == (('f'<<8) | 'L'))
		USEX(SCI_MODE) |= (1<<SCIMB_OUT_OF_WAV);
#endif
	    }
	  }
#endif
#endif
	}
    }
#ifdef NO_WMA /* reject WMA files */
    if (USEX(SCI_HDAT1) == (('W'<<8)|'M')) {
      USEX(SCI_AICTRL0) += 1; /* next file */
      goto goreset;//MyReset();
    }
#endif
#if defined(NO_AAC) && !defined(SCI_UI) && !defined(RECORDER)
    {
      register u_int16 t = USEX(SCI_HDAT1);
      //0x4154(AT) 0x4144(AD) 0x4d34(M4)
      if (t == 0x4154 || t == 0x4144 || t == 0x4d34) {
	USEX(SCI_AICTRL0) += 1; /* next file */
	goto goreset;//MyReset();
      }
    }
#endif
  } else if (cachedSpiState == spi_zeros) {
    /* Workaround for WMA resync after stream ended! */
    parametric_x.resync = 0;
    
    if (StreamDiff() < STREAM_BUFFER_SZ-40) {
      register int i, z = parametric_x.endFillByte;
      register u_int16 *wr = (void *)stream_wr_pointer;
      for (i=0;i<32;i++) {
	*wr = z;
	if (wr >= &stream_buffer[STREAM_BUFFER_SZ-1]) {
	  wr = stream_buffer;
	} else {
	  wr++;
	}
	++spiCnt;
	stream_wr_pointer = (void *)wr;
      }
      /* watchdog clearing here doesn't work with midi */
#ifdef TIMEOUT
      timeOut = 0;
#endif
    }
#ifdef TIMEOUT
    if ((s_int16)++timeOut < 0) {
      //putch(USEX(SCI_HDAT1)>>8);
      //putch(USEX(SCI_HDAT1));
      goto goreset;
    }
#endif
    
    if (USEX(SCI_HDAT1) == 0 ||
#if 1
	USEX(SCI_HDAT1) == 0x574d ||
#endif
	spiCnt >= 256+2048) { /* was left to max 256 by spi_data */
      /* 2048 words instead of bytes */
    goreset:
      /* Discard any unread data, including CRC, before starting again */
      {
	register int i;
	for (i=0; i<256+2; i++)
	  SpiSendReceiveMmc(-1/*0xffff*/,16);
      }
      SpiSendClocks();

#ifdef UART_UI
      putch('e');//USEX(UART_DATA) = 'e';
#endif
      MyReset();
    }
#ifdef ENABLE_WATCHDOG
    goto wdogack;//USEX(WDOG_RESET) = WDOG_RESET_VAL;
#endif
  } else {
#ifdef ENABLE_WATCHDOG
  wdogack:
    USEX(WDOG_RESET) = WDOG_RESET_VAL;
#endif
  }
#endif /* VOICE_ACTIVATED*/
    
#ifdef SCI_UI
  USEX(SER_DREQ) = 0;
#else
    
#ifndef NO_DREQ_LED
  /* 15 words */
  {
    register int t = 0;
    
    if ((ui.Timer & 0x18) == 0) {
      /* DREQ indicates random play status */
#ifdef NO_RANDOM
      t = USEX(SCI_BASS); /* has lowest bit set if enabled*/
#else
      t = USEX(SCI_AICTRL3);  /* lowest bit is global random play */
#ifdef LOUDNESS_ALWAYS_ON_OFF
      /* indicate random play */
#else /*LOUDNESS_ALWAYS_ON_OFF*/
                  
      if (cachedSpiState
#ifdef SW3_RANDOM_PLAY
	  ==
#else
	  !=
#endif
	  spi_pause)
	t = USEX(SCI_BASS); /* has lowest bit set if enabled*/
#endif /*else LOUDNESS_ALWAYS_ON_OFF*/
#endif /*else NO_RANDOM*/
    }
    USEX(SER_DREQ) = t;
  }
#endif/*!NO_DREQ_LED*/
#endif/*!SCI_UI*/
    
  spiState = cachedSpiState;

}


#ifdef VS1053
void NewSinTest(void);
auto int PatchFrame(void);
#endif


auto void MyMain(register s_int16 __a1 fullReset) {

 


  fileNamePos = -20; //File name position on screen


  USEX(DECIM_CONTROL) = DECIM_ENABLE | DECIM_FACTOR48K; // Decimator on, factor, enabled.
  USEX(0xC000) &= ~(0x00c0); //Analog power up
  USEX(SCI_STATUS) &= ~((1<<SCIST_APDOWN1) | (1<<SCIST_APDOWN2)); /* enable ANALOG power */
  

  USEX(INT_ENABLE) = ENABLE_SCI_DAC_RX;
  USEX(SCI_DECODE_TIME) = 0;

#ifdef ENABLE_HIGHREF
  applAddr = ApplAddrRef;
  USEX(SCI_STATUS) |= (1 << SCIST_REFERENCE_SEL);
#endif

  
#if 0 && defined(VS1053) //Sine Test
  applAddr = ApplAddrRef;
  //ScreenInit();
  //ScreenLocate (50,4);
  //ScreenPutPacked("\pSine Test",100);
  USEX(SCI_VOL) = 0x1010;
  SetRate(44100U);
  USEX(SCI_AICTRL0) = 1972; /*1kHz  2972*/
  USEX(SCI_AICTRL1) = 1972;
  while(1) {
    USEX(WDOG_RESET) = WDOG_RESET_VAL;
    CallIROM4(NewSinTest);
  }
#endif
      
  ScreenInit();
  ScreenLocate(5,5);
  ScreenPutPacked("\pInit.",128);
  USEX(SER_DREQ) = 0; /* Led / Aux key scan off*/
  do { //Wait until power key is NOT pressed;
    USEX(WDOG_RESET) = WDOG_RESET_VAL;
  } while ((USEX(GPIO_IDATA) & (1 << 10)) != 0);
      
      
#ifdef UART_BUFFERED
  UartFill();
#endif

#ifdef VS1053
  /* Switch */
  USEX(VS1053_IROM4) &= ~(IROM4_ENABLE);
#endif

#if defined(SCI_UI) && defined(ENABLE_WATCHDOG)
  USEX(WDOG_CONFIG) = 1000;
  USEX(WDOG_RESET) = WDOG_RESET_VAL;
#endif

#if defined(SCI_UI)
  USEX(SCI_AICTRL1) = 0;
#endif

  USEX(SCI_MODE) = SciModeOr() | (USEX(SCI_MODE) & ~(1<<SCIMB_OUT_OF_WAV));
  USEX(GPIO_ODATA) = SPI_xCS|SPI_CLK|SPI_xCS2|(1<<LCD_GPIO_PIN_A0)|(1<<LCD_GPIO_PIN_CS); /*use same= -1 word*/
  USEX(GPIO_DDR)   = SPI_xCS|SPI_CLK|SPI_xCS2|(1<<LCD_GPIO_PIN_A0)|(1<<LCD_GPIO_PIN_CS);
#ifdef VOICE_ACTIVATED
  | (1<<7)
#endif        
#if defined(VS1053) && defined(ENABLE_I2S)
      /* Enable I2S */
      | 0xf0;
  USEX(I2S_CONTROL) = 0x0c
#endif
    ;

#ifdef VOICE_ACTIVATED
  putch('3');
#endif

        
  InitHardware(); /* no longer clears memory in vs1053 */
#ifdef RECORDER
  record.record <<= 1; /* 0x1234 to 0x2468 */
#endif
    
#ifndef USE_DEBUG
  /* Allow reflashing without MMC/SD inserted. */
  USEX(INT_ENABLE) = (1<<INT_EN_RX);
#endif
  InitLayer3();
  InitCommon();
  InitDecode();

    
#if !defined(ENABLE_LAYER12)
  /* If not powerup reset, re-enable analog immediately. */
  if (!fullReset) {
    USEX(SCI_STATUS) &= ~(1<<SCIST_APDOWN2);
  }
#if defined(ENABLE_WATCHDOG) && defined(KEEP_DRIVERS_OFF)
  else {
    /*analog power off and analog driver powerdown */
    //USEX(SCI_STATUS) |= (1<<SCIST_APDOWN1)|(1<<SCIST_APDOWN2);
  }
#endif
#endif
    
    
#if defined(SCI_UI) && defined(VS1053) && defined(ENABLE_HIGHREF)
  applAddr = ApplAddrRef;
#endif
#ifdef ENABLE_UI
#if defined(VS1053) && defined(ENABLE_HIGHREF)
  applAddr = ApplAddrButRef;
#else
  applAddr = ApplAddr;
#endif
#endif/*ENABLE_UI*/
    
   
  /* Allow DREQ and enable interrupts, VERY IMPORTANT! */
  USEX(INT_GLOB_ENA) = 0;
  USEX(INT_GLOB_ENA) = 0;

  InitReadFrame();

#ifdef USE_LCD_DISPLAY
  SpiSendClocks();
  ScreenInit();
  SpiSendClocks();
#endif

  DecodeTimeClear();

  /* while in USB */
  if (USEX(GPIO_IDATA) & (1 << GPIO_PIN_USB_DETECT)){    
    ScreenInit();	
    ScreenLocate(50,4);
    ScreenPutBigChar('U');
    ScreenPutBigChar('S');
    ScreenPutBigChar('B');
    while (USEX(GPIO_IDATA) & (1 << 9)){	
      ScreenPutVolts();
    }
  }
    
  /* wake up MMC! -- If this takes too long, watchdog will bite. */

#ifdef SAVE_POSITION
  if (USEX(SCI_AICTRL0)) /* 9 words */
    SaveWord(28/*AICTRL0*/, 0); //3 words
#endif
    

  //while (1)
  {
    static u_int16 retries = 5;
    register s_int16 i, cmd;
    s_int32 d;
      
    // Start the MMC/SD card
  tryagain: 

    // MMC reset
    USEX(GPIO_DDR) |= (1 << GPIO_PIN_SDCARD_OFF)/*SO out*/;
      
      
    USEX(SER_DREQ) = 0;
    if (++retries>=5) {
      u_int16 gpio0_save = USEX(GPIO_ODATA);
      retries = 0;
      USEX(GPIO_ODATA) = 1; //all 0 except eecs
      for (d=0; d<320000; d++) {
	USEX(GPIO_ODATA) = 1 + (1 << GPIO_PIN_SDCARD_OFF/*SO*/); //all 0 except eecs
      }
      USEX(GPIO_ODATA) = gpio0_save;
    }
    USEX(GPIO_ODATA) &= ~(1 << GPIO_PIN_SDCARD_OFF); //SD ON
      
    for (i=0; i<512; i++) {
      SpiSendClocks();
    }
      
      
    /* MMC Init, command 0x40 should return 0x01 if all is ok. */
    i = MmcCommand(MMC_GO_IDLE_STATE/*CMD0*/|0x40,0);
    if (i != 1)
      goto tryagain;//continue; /* No valid idle response */
      
    cmd = MMC_SEND_OP_COND|0x40;
#ifdef USE_HC
    //MmcCommand(MMC_CRC_ON_OFF/*CMD59*/|0x40, 0);
    if (MmcCommand(MMC_SEND_IF_COND/*CMD8*/|0x40, 0x00000122/*2.7-3.6V*/)
	== 1) {
      /* MMC answers: 0x05 SD answers: 0x01 */
      /* Should we read the whole R7 response? */
      //SpiSendReceiveMmc(-1, 32);
      cmd = 0x40|41; /* ACMD41 - SD_SEND_OP_COND */
    }
#endif /*USE_HC*/
	
    /* MMC Wake-up call, set to Not Idle (mmc returns 0x00)*/
    //i = 0; /* i is 1 when entered .. but does not make the code shorter*/
    while (1) {
      register int c;
#ifdef USE_HC
      if (cmd == (0x40|41)) {
	MmcCommand(0x40|55/*CMD55*/,0);
	c = MmcCommand(cmd/*MMC_SEND_OP_COND|0x40*/, 0x40000000UL);
      } else
#endif
	c = MmcCommand(cmd/*MMC_SEND_OP_COND|0x40*/, 0);
      if (c == 0)
	break;
      if (++i >= 12000/*< 0*/ || c != 1) {
	goto tryagain; /* Not able to power up mmc */
      }
    }
	
#ifdef USE_HC
    hcShift = 9;
    i = 0x0246;
    /* PRETEC's 1G MMC does not seemd to handle CMD58 here! */
    /* Support HC for SD, but not for MMC! */
    if (cmd == (0x40|41) &&
	MmcCommand(MMC_READ_OCR/*CMD58*/|0x40, 0) == 0) {
      if (SpiSendReceiveMmc(-1, 16) & (1<<(30-16))) {
	/* OCR[30]:CCS - card capacity select */
	/* HighCapacity! */
	/* ldc 9,D0 (0x0246) <-> ldc 0,D0 (0x0006) */
	i = 6;
	hcShift = 0;
      }
      //SpiSendReceiveMmc(-1, 16);
    }
    WriteToProgramRam((u_int16)ReadDiskSector+2, 0, i);
#endif /*USE_HC*/
        
#if 0
    /* Set Block Size of 512 bytes -- default for at least HC */
    /* Needed by MaxNova S043618ATA 2J310700 MV016Q-MMC */
    /* Must check return value! (MicroSD restart!) */
    if (MmcCommand(MMC_SET_BLOCKLEN|0x40, 512) != 0)
      goto tryagain;
    /* All OK return */
#endif
  }

  // Init File System (FAT)

  /* Detects cards without partition table */
  if (InitFileSystem() == 0) {

        
#ifdef SCI_UI
    if ((USEX(SCI_AICTRL3) & CTRL3_NO_NUMFILES)) {
      USEX(SCI_AICTRL1) = 0x7fff;
    } else
#endif/*SCI_UI*/
      {
	/* determine number of playable files */
	USEX(SCI_AICTRL1) = OpenFile(0xffffU);
	//putch(USEX(SCI_AICTRL1)>>8);
	//putch(USEX(SCI_AICTRL1));
      }
	
#if !defined(NO_RANDOM) /* if random play feature active */
    if ((USEX(SCI_AICTRL3) & CTRL3_RANDOM_PLAY)) {
#ifdef SHUFFLE_PLAY /*shuffle 23 words*/
      USEX(SCI_AICTRL0) = Shuffle(USEX(SCI_AICTRL1), USEX(SCI_AICTRL0));
#else
      /* random play (not shuffle!) */
      USEX(SCI_AICTRL0) = (myrand() % USEX(SCI_AICTRL1));
#endif
    }
#endif
        
    if (OpenFile(USEX(SCI_AICTRL0)
#ifdef SCI_UI
		 &= 0x7fffU
		 //max 32768 files in SCI-controlled version
#endif
		 ) < 0) {
#if defined(SKIP_ID3V2) /* 48 words */
      /* check for ID3v2 */
      /* An ID3v2 tag can be detected with the following pattern:
	 $49 44 33 yy yy xx zz zz zz zz */
      ReadDiskSector(fragments[0].start);
      if ((GetLong(0) & 0xffffffL) == 0x334449) {
	/* skip in 16kB resolution, assume tags are smaller than 2MB */
	/* 0.5kB resolution now */
	register u_int16 skip = 32 * GetByte(7) + GetByte(8)/4;
	/* guard against small fragments */
	if (skip >= fragments[0].size)
	  skip = fragments[0].size-1;
	fragments[0].start += skip;
	fragments[0].size  -= skip;
	fatInfo.fileSize -= (u_int32)skip * 512;
      }
#endif
            
#if defined(SCI_UI)
      if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) ==
	  CTRL3_PAUSE_BEFORE_PLAY) {
	/* Mark file ready and go to pause state in UserHook */
	USEX(SCI_AICTRL3) |= CTRL3_PAUSE_ON;
      }
      /*NEW: set FILE_READY always */
      USEX(SCI_AICTRL3) = CTRL3_FILE_READY | 
	(USEX(SCI_AICTRL3) & ~CTRL3_AT_END);
#endif
            
#define PAUSE_AT_POWERON
#if defined(PAUSE_AT_POWERON) /*pause at power-on, then use pause after play*/
      if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) ==
	  CTRL3_PAUSE_BEFORE_PLAY) {
	USEX(SCI_AICTRL3) = CTRL3_PAUSE_AFTER_PLAY;
	oldSpiState = spi_file;
	spiState = spi_pause;
	stream_save = stream_wr_pointer;
      } else {
	spiState = spi_file;
      }
#else /*PAUSE_AT_POWERON*/
            
#if defined(UI_PAUSE_BEFORE_PLAY) && defined(ENABLE_UI)
      oldSpiState = spi_file;
      spiState = spi_pause;
      stream_save = stream_wr_pointer;
#else
      spiState = spi_file;
#endif
#endif /*else PAUSE_AT_POWERON*/
            
            
#ifdef HDATCHECK
      hdatCheck = 0;
#endif
    } else {
    gofirst:
      USEX(SCI_AICTRL0) = 0; /* file not found, jump to first file */
      MyReset();
    }
  } else {
    /* Initialize again */
    goto gofirst;//MyReset();
  }

      
#if !defined(RECORDER) && defined(ENABLE_WATCHDOG) && defined(KEEP_DRIVERS_OFF)
  /*otherwise keep it in powerdown until MMC found*/
  USEX(SCI_STATUS) &= ~(1<<SCIST_APDOWN1); /* enable ANALOG power */
#endif
      
      
#ifdef SAVE_POSITION
  SaveWord(28/*AICTRL0*/, USEX(SCI_AICTRL0));
#endif

  USEX(INT_ENABLE) = ENABLE_SCI_DAC_RX;
  USEX(SCI_DECODE_TIME) = 0;

  parametric_x.resync = 32767;

#if defined(FIX_WAV_DECODETIME) && defined(SCI_UI)
  lastDecodeTime = 0;
#endif /*SCI_UI and FIX_WAV_DECODETIME*/
    
  stream_rd_pointer = stream_wr_pointer = stream_buffer;
  stream_rd_odd = 0;
      

      
  while (1) {
    if (
#if !defined(RECORDER) && defined(USE_FLAC)
	!PatchFrame()
#else
	!ReadFrame()
#endif
	) {
      play_frame();
    } else {
      InitReadFrame();
    }
  }
}
