
/*

FILE: prom.c - SPI boot programmer
DESCRIPTION: 

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define STANDALONE
#include "h1053/hardware.h"

#undef puts
#undef putchar

void putch(register __a0 short ch);

void Halt();
__near void vs3emubreak(register __i1 bp);


typedef long s_int32;
typedef short s_int16;
typedef unsigned short u_int16;
typedef unsigned long u_int32;
typedef __fract short f_int16;
typedef __fract long f_int32;

int largeAddr = 0;

static char hex[] = "0123456789abcdef";
void puthex(unsigned int a) {
    char tmp[5];
    tmp[0] = hex[(a>>12)&15];
    tmp[1] = hex[(a>>8)&15];
    tmp[2] = hex[(a>>4)&15];
    tmp[3] = hex[(a>>0)&15];
    tmp[4] = '\0';
    puts(tmp);
}

//#define USEY(x) *((__y volatile unsigned short *)(unsigned short)(x))
#ifndef GPIO_IDATA
#define GPIO_IDATA GPIO_DATA
#endif

#define SPI_xCS   1 /* GPIO */
#define SPI_CLK   2
#define SPI_MISO  4
#define SPI_xCS2  8
#define SPI_MOSI  1 /* DREQ */

#define SPI_WREN  0x06
#define SPI_WRDI  0x04
#define SPI_RDSR  0x05
#define SPI_WRSR  0x01
#define SPI_READ  0x03
#define SPI_WRITE 0x02

auto void SpiPrivDelay(void) {
    int i;
    for (i=0;i<10;i++) {
	USEX(SER_DREQ);
    }
}
auto u_int16 SpiPrivSendReceive(register __a0 u_int16 dat,
			    register __a1 s_int16 bits) {
    register u_int16 odata = 0;
    while (bits--) {
	USEX(SER_DREQ) = (dat>>bits);
	SpiPrivDelay();
	USEX(GPIO_ODATA) |= SPI_CLK;
	SpiPrivDelay();
	odata <<= 1;
#if SPI_MISO == 4
	odata |= (USEX(GPIO_IDATA) >> 2) & 1;
#else
	odata |= (USEX(GPIO_IDATA) & SPI_MISO) ? 1 : 0;
#endif
	USEX(GPIO_ODATA) &= ~SPI_CLK;
    }
    return odata;
}


void SpiPrivInit(void) {
    USEX(GPIO_ODATA) = SPI_xCS|SPI_xCS2;
    USEX(GPIO_DDR) = SPI_xCS|SPI_CLK|SPI_xCS2;
    USEX(GPIO_ODATA) = SPI_xCS;
}

u_int16 SpiPrivRead(u_int16 addr) {
    u_int16 dat;

    USEX(GPIO_ODATA) |= SPI_xCS;
    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_READ, 8);
    SpiPrivDelay();
    if (largeAddr) {
	SpiPrivSendReceive(0, 8); //24-bit address
    }
    SpiPrivSendReceive(addr, 16);
    SpiPrivDelay();
    dat = SpiPrivSendReceive(0, 16);
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;
    return dat;
}

u_int16 SpiPrivStatus(void) {
    u_int16 dat;

    USEX(GPIO_ODATA) |= SPI_xCS;
    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_RDSR, 8);
    dat = SpiPrivSendReceive(0, 8);
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;
    return dat;
}

u_int16 SpiPrivWrite(u_int16 addr, u_int16 wrdat) {
    u_int16 dat;

    USEX(GPIO_ODATA) |= SPI_xCS;
    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_WREN, 8);
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;

    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_WRITE, 8);
    SpiPrivDelay();
    if (largeAddr) {
	SpiPrivSendReceive(0, 8); //24-bit address
    }
    SpiPrivSendReceive(addr, 16);
    SpiPrivDelay();
    dat = SpiPrivSendReceive(wrdat, 16);
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;
    return dat;
}

u_int16 SpiStatus(void) {
    register u_int16 dat;

    USEX(GPIO_ODATA) |= SPI_xCS;
    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_RDSR, 8);
    dat = SpiPrivSendReceive(0, 8);
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;
    return dat;
}


#if 1
#define EEPROM_PAGE_SIZE_BITS 5
#define EEPROM_PAGE_SIZE (1<<EEPROM_PAGE_SIZE_BITS)

u_int16 SpiBlockWrite(u_int16 addr, register u_int16 *buf, u_int16 len) {
    USEX(GPIO_ODATA) |= SPI_xCS;
    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_WREN, 8);
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;


    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_WRITE, 8);
    if (largeAddr) {
	SpiPrivSendReceive(0, 8); //24-bit address
    }
    SpiPrivSendReceive(addr, 16);
    while (len--) {
	SpiPrivSendReceive(*buf, 16);
	addr += 2;
	buf++;
    }
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;
    return addr;
}

u_int16 SpiVerify(u_int16 addr, register u_int16 *buf, u_int16 len) {
    u_int16 dat;

    USEX(GPIO_ODATA) |= SPI_xCS;
    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_READ, 8);
    SpiPrivDelay();
    if (largeAddr) {
	SpiPrivSendReceive(0, 8); //24-bit address
    }
    SpiPrivSendReceive(addr, 16);
    SpiPrivDelay();

    while (len--) {
	if (SpiPrivSendReceive(0, 16) != *buf++) {
	    SpiPrivDelay();
	    USEX(GPIO_ODATA) |= SPI_xCS;
	    return -1;
	}
    }
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;
    return 0;
}

u_int16 BlockProgram(u_int16 addr, u_int16 *buf, s_int16 words) {
    register s_int16 i, len = words;
    for (i=0;i<len;) {
	register u_int16 sz = (((addr&~(EEPROM_PAGE_SIZE-1))+EEPROM_PAGE_SIZE)
			       - addr) / 2;
	if (sz > len-i)
	    sz = len-i;
	SpiBlockWrite(addr, buf+i, sz);
	i += sz;
	addr += 2*sz;
	while (SpiPrivStatus() & 1)
	    SpiPrivDelay();
    }
    return SpiVerify(addr - 2*len, buf, len);
}
#endif

/*

FUNCTION: main()
DESCRIPTION: RTOS, peripheral and variable initialization.
INPUTS: none
RESULT: none
SEE ALSO: 

*/
auto void SpiSendClocks(void);

#define BUF_SIZE 256
void MyMain(void) {
    FILE *fp;
    static u_int16 buf[BUF_SIZE];
    int i, len, addr = 0;

    puthex(USEX(UART_DIV));
    puts(" div");

    USEX(INT_ENABLE) = 0;
    SpiPrivInit(); /* Set SPI clock etc */

#if 1
    for(i=0;i<512;i++)
	SpiSendClocks();
#endif

#if 0
    while (1)
	SpiStatus();
#endif

    puthex(SpiStatus()); /* should be 0080 */
    puts(" status");
#if 1
    while (addr < 40/*32766U*/) {
	len = SpiPrivRead(addr);
	puthex(len);
	addr += 2;
    }
    addr = 0;
#endif
#if 0
    USEX(GPIO_ODATA) |= SPI_xCS;
    SpiPrivDelay();
    USEX(GPIO_ODATA) &= ~SPI_xCS;
    SpiPrivSendReceive(SPI_WREN, 8);
    SpiPrivDelay();
    USEX(GPIO_ODATA) |= SPI_xCS;
    puthex(SpiStatus()); /* should be 0082 */
    puts(" status");
    while (addr < 32766U) {
	len = SpiPrivRead(addr);
	puthex(len);
	addr += 2;
    }
#endif

#if 0
    if (SpiPrivRead(0) == 0x5026 && (SpiPrivRead(2)&0xff00U) == 0x4800) {
	addr = 3;
	while (1) {
	    u_int16 com = (SpiPrivRead(addr)>>8);
	    u_int16 len = SpiPrivRead(addr+1);
	    u_int16 add = SpiPrivRead(addr+3);
	    puthex(com);
	    putchar(' ');
	    putchar('l');
	    puthex(len);
	    putchar(' ');
	    putchar('a');
	    puthex(add);
	    putchar('\n');
	    addr += len+5;
	    if (com == 3)
		break;
	}
    }
    addr = 0;
/*
X: 0xc002-0xc00d In:   27, out:   27
I: 0x0000-0x0002 In:   13, out:   13
I: 0x0030-0x0455 In: 4249, out: 4249
*/
#endif


#if 0
    SpiPrivWrite(0, 0xaaaa);
    SpiPrivRead(0x1111);
    while (1)
	;
#endif
#ifdef VS1002
    fp = fopen("boot1002.img", "rb");
#else
    fp = fopen("boot.img", "rb");
#endif
    if (!fp) {
	puts("Could not open boot.img!");
	goto exit;
    }
    while (len = fread(buf, 1, BUF_SIZE, fp)) {
#ifdef EEPROM_PAGE_SIZE
	if (BlockProgram(addr, buf, len)) {
	    putchar('x');
	    putchar('\n');
	} else {
	    putchar('.');
	    fflush(stdout);
	}
	addr += len*2;
#else
	for (i=0;i<len;i++) {
	    SpiPrivWrite(addr, buf[i]);
#if 0
	    printf("0x%04x\n", SpiPrivStatus());
#endif
	    if (!(addr & 15)) {
		putchar('.');
		fflush(stdout);
	    }
	    while (SpiPrivStatus() & 1) {
		SpiPrivDelay();
	    }

#if 0
	    puthex(SpiPrivRead(addr));
#endif
#if 1
	    if (buf[i] != SpiPrivRead(addr)) {
		puthex(addr);
		putchar(' ');
		puthex(SpiPrivRead(addr));
		putchar('\n');
	    }
#endif
	    addr += 2;
	}
#endif
    }
    fclose(fp);

    puts("\nFinished!!\n");
    fflush(stdout);


#if 0
    addr = 1536;

    fp = fopen("data.img", "rb");
    if (!fp) {
	puts("Could not open data.img!");
	while (1)
	    ;
    }
    while (len = fread(buf, 1, BUF_SIZE, fp)) {
	for (i=0;i<len;i++) {
	    SpiPrivWrite(addr, buf[i]);
	    if (!(addr & 15)) {
		putchar('.');
		fflush(stdout);
	    }
	    while (SpiPrivStatus() & 1)
		SpiPrivDelay();
	    addr += 2;
	    if (addr >= 8192)
		break;
	}
	if (addr >= 8192)
	    break;
    }
    fclose(fp);
    puts("\nFinished!!\n");
    fflush(stdout);
#endif

#if 1
 exit:
    {
	int i;
	for (i=0;i<61;i++)
	    putch(0x10);
	while ((USEX(UART_STATUS) & UART_ST_TXRUNNING))
	    ;
    }
#endif
    ((void (*)(void))(0x4000))();
}
