#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>
#include "tedaxbooktovsdsp.h"
#include "audioinfo.h"

/*
  Very simple and crude tEDAx to VSDSP converter example.

  http://repo.hu/projects/tedax/
 */

#define VERSION "1.03"

#if 0
#define DEBUG
#endif

#if 1
#define USE_STRING_COMPRESSION
#endif

#define BIN_BUFFER_SIZE (16*1024*1024)
#define STRING_BUFFER_SIZE (16*1024*1024)
#define POINTER_BUFFER_SIZE (16*1024*1024)

/* Big buffer for processed binary tEDAx representation. */
char binBuffer[BIN_BUFFER_SIZE];
int binBufferUsed = 0;
/* String buffer for all tEDAx strings. */
char stringBuffer[STRING_BUFFER_SIZE];
int stringBufferUsed = 0;
/* Pointer indices in binBuffer */
int pointers[POINTER_BUFFER_SIZE];
int pointersUsed = 0;


struct TedaxBook book;
struct TedaxLevelItem levelItem[MAX_TEDAX_LEVEL_ITEMS];
struct TedaxLevelItem *lastLevelItem[MAX_TEDAX_LEVELS+1];
int levelItems;
int itemsPerLevel[MAX_TEDAX_LEVELS+1], levelOffset[MAX_TEDAX_LEVELS+1];
int addIdentifiers = 1;


/* Inserts a raw 16-bit value into the bit buffer.
   No range checking is made. */
void InsertRaw16(int value, int pointer) {
  binBuffer[pointer+0] = (value >> 8) & 0xFF; /* Big-endian MSB */
  binBuffer[pointer+1] = (value & 0xFF);      /* Big-endian LSB */
}

/* Range checks and inserts an unsigned 16-bit value into the bit buffer. */
void InsertUInt16(int value, int pointer) {
  if (value < 0) value = 0;
  if (value > 65535) value = 65535;
  InsertRaw16(value, pointer);
}

/* Range checks and inserts a signed 16-bit value into the bit buffer. */
void InsertSInt16(int value, int pointer) {
  if (value < 0) value = 0;
  if (value > 65535) value = 65535;
  InsertRaw16(value, pointer);
}

/* Range checks and adds an unsigned 16-bit value to end of bit buffer. */
void AddUInt16(int value) {
  assert(binBufferUsed+2 <= BIN_BUFFER_SIZE);
  InsertUInt16(value, binBufferUsed);
  binBufferUsed += 2;
}

/* Range checks and adds a signed 16-bit value to end of bit buffer. */
void AddSInt16(int value) {
  assert(binBufferUsed+2 <= BIN_BUFFER_SIZE);
  InsertSInt16(value, binBufferUsed);
  binBufferUsed += 2;
}

/* Inserts a signed 32-bit value into the bit buffer.
   No range checking is made. */
void InsertSInt32(int value, int pointer) {
  InsertRaw16( value        & 0xFFFF, pointer  ); /* Mixed-endian LSW */
  InsertRaw16((value >> 16) & 0xFFFF, pointer+2); /* Mixed-endian MSW */
}

/* Adds a signed 32-bit value to the end of the bit buffer.
   No range checking is made. */
void AddSInt32(int value) {
  assert(binBufferUsed+4 <= BIN_BUFFER_SIZE);
  InsertSInt32(value, binBufferUsed);
  binBufferUsed += 4;
}

/* Gets an unsigned 16-bit value from the bit buffer */
s_int32 GetUInt16(int pointer) {
  /* Return big-endian 16-bit value */
  return (binBuffer[pointer] << 8) | binBuffer[pointer+1];
}

/* Gets a signed 32-bit value from the bit buffer */
s_int32 GetSInt32(int pointer) {
  /* Return mixed-endian 32-bit value */
  s_int32 retVal = GetUInt16(pointer);              /* Mixed-endian LSW */
  retVal |= ((s_int32)GetUInt16(pointer+2)) << 16;  /* Mixed-endial MSW */
  assert(sizeof(retVal) == 4);
  return retVal;
}

/* Adds a 32-bit string pointer to the end of the bit buffer.
   If pointer < 0, then make it point to the end of string buffer. */
void AddStringPointer(s_int32 ptr) {
  assert(pointersUsed < POINTER_BUFFER_SIZE);
  pointers[pointersUsed++] = binBufferUsed;
  AddSInt32((ptr < 0) ? stringBufferUsed : ptr);
}

/* Adds a string to the end of the string buffer. Also adds a 32-bit string
   pointer to the end of the bit buffer, or a NULL pointer if no string is
   present. */
void AddString(char *s) {
  int lenWithNul = strlen(s)+1;       /* Trailing NUL not included. */
  if (lenWithNul > 1) {
#ifdef USE_STRING_COMPRESSION
    /* If string compression is used, similar strings are looked for.
       This will help significantly with e.g. repeated file names.

       Note: This search method is o(n^2) slow, but running on a fast
       PC it doesn't matter. */
    char *sP = stringBuffer;
    while (sP < stringBuffer+stringBufferUsed) {
      if (!strcmp(s, sP)) {
	AddStringPointer(sP-stringBuffer);
	goto finally;
      }
      sP += strlen(sP)+1;
    }
#endif
    assert(stringBufferUsed+lenWithNul <= STRING_BUFFER_SIZE);
    AddStringPointer(-1);  
    memcpy(stringBuffer+stringBufferUsed, s, lenWithNul);
    stringBufferUsed += lenWithNul;
  } else {
    AddSInt32(0); /* Null pointer */
  }
 finally:
  return;
}



char *mystrtok(char *str, const char *delim) {
  char *p = strtok(str, delim);
  if (!p) return NULL;
#if 0
  while (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n') {
    p++;
  }
#endif
  return p;
}

void Unescape(char *s) {
  char *wp = s;
  int unescape = 0;
  while (*s) {
    if (unescape) {
      *wp++ = *s;
      unescape = 0;
    } else if (*s == '\\') {
      unescape = 1;
    } else {
      *wp++ = *s;
    }
    s++;
  }
  *wp++ = '\0';
}




/* Different contexts for the tEDAx parser. */
enum Context {
  ctNone = 0,
  ctTedax,
  ctBook,
  ctLevelItem,
};



/* Reads the tEDAx file and parses it to internal buffers */
int ReadTedaxFile(FILE *fp) {
  int errCode = -1;
  char s[MAX_TEDAX_LINE_LENGTH];
  char sCopy[MAX_TEDAX_LINE_LENGTH];
  int lineNo = 0, i;
  enum Context context = ctNone;
  int currLevel = -1;
  char lastAudioFileName[MAX_TEDAX_LINE_LENGTH] = "";
  s_int32 lastAudioFileLenMS = -1;

  
  /* For each line of input */
  while (!feof(fp)) {
    int len;
    char *p;
    lineNo++;
    if (!(fgets(s, MAX_TEDAX_LINE_LENGTH, fp))) break;
    len = strlen(s);
    if (len) {
      if (s[len-1] != '\n') {
	fprintf(stderr, "%d: \"%s\" too long or doesn't end in newline\n",
		lineNo, s);
	goto finally;
      }
      s[len-1] = '\0';
    }

    /* Make a copy of the line because strtok() will break the origal */
    strcpy(sCopy, s);

    /* Begin tokenizing sCopy */
    p = strtok(sCopy, " \t\n\r");

    /* If line exists and it does not start with comment char '#' */
    if (p && p[0] != '#') {
      switch(context) {
      case ctNone:
	/* No context: tEDAx line is the only one that is accepted */
	if (!strcmp(p, "tEDAx")) {
	  p = strtok(NULL, " \t\n\r");
	  if (!strcmp(p, "v1") && !(p = strtok(NULL, " \t\n\r"))) {
	    context = ctTedax;
	  } else {
	    fprintf(stderr, "%d: incorrect tEDAx line: \"%s\"\n", lineNo, s);
	    goto finally;
	  }
	} else {
	  fprintf(stderr, "%d: expecting tEDAx line: \"%s\"\n", lineNo, s);
	  goto finally;
	}
	break;
      case ctTedax:
	if (!strcmp(p, "begin")) {
	  char *p2[3];
	  /* Parse e.g. (begin) book v1 The\ Bible */
	  p2[0] = strtok(NULL, " \t\n\r");
	  p2[1] = strtok(NULL, " \t\n\r");
	  p2[2] = strtok(NULL, "");
	  Unescape(p2[2]);
	  if (!p2[2] || strcmp(p2[1], "v1")) {
	    fprintf(stderr, "%d: incorrect begin line: \"%s\"\n", lineNo, s);
	    goto finally;
	  }
	  if (!strcmp(p2[0], "book")) {
	    context = ctBook;
	    memset(&book, 0, sizeof(book));
	    strcpy(book.name, p2[2]);
	  } else if (!strcmp(p2[0], "levelitem")) {
	    context = ctLevelItem;
	    memset(levelItem+levelItems, 0, sizeof(levelItem[0]));
	    strcpy(levelItem[levelItems].ident, p2[2]);
	  } else {
	    fprintf(stderr, "%d: unknown type in begin line: \"%s\"\n", lineNo, s);
	    goto finally;
	  }
	} else {
	  fprintf(stderr, "%d: unexpected line: \"%s\"\n", lineNo, s);
	  goto finally;
	}
	break;
      case ctBook:
	if (!strcmp(p, "levelname")) {
	  char *p2[2];
	  int levelNumber;
	  p2[0] = strtok(NULL, " \t\n\r");
	  p2[1] = strtok(NULL, "");
	  if (!p2[1] ||
	     (levelNumber = atoi(p2[0])) < 0 ||
	     levelNumber > MAX_TEDAX_LEVELS) {
	    fprintf(stderr, "%d: incorrect levelname line: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }
	  strcpy(book.levelName[levelNumber], p2[1]);
	} else if (!strcmp(p, "end")) {
	  char *p2[2];
	  p2[0] = strtok(NULL, " \t\n\r");
	  p2[1] = strtok(NULL, "");
	  if (p2[1] || !p2[0] || strcmp(p2[0], "book")) {
	    fprintf(stderr, "%d: incorrect end line inside book: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }
	  context = ctTedax;

	  /*
	    Write book structure into binBuffer
	  */
	  /* Identifier */
	  AddSInt32(BOOK_IDENTIFIER);     /* VSDSP: book.identifier */
	  AddSInt16(BOOK_VERSION_NUMBER); /* VSDSP: book.version */
	  /* Number of level items will be updated at emit stage.
	     Offset is 6, do NOT change number of fields before this one!*/
	  AddSInt16(0x0);                 /* VSDSP: book.levelItems */
	  AddString(book.name);           /* VSDSP: book.nameP */
#ifdef DEBUG
	  printf("book %s\n", book.name);
#endif
	  for (i=0; i<MAX_TEDAX_LEVELS; i++) {
#ifdef DEBUG
	    printf("Level %d: \"%s\"\n", i, book.levelName[i]);
#endif
	    AddString(book.levelName[i]); /* VSDSP: book.levelNameP[i] */
	  }
	  AddSInt32(0x0); /* book.padding[0] */
	  AddSInt32(0x0); /* book.padding[1] */
	  AddSInt32(0x0); /* book.padding[2] */
	  AddSInt32(0x0); /* book.padding[3] */
	} else {
	  fprintf(stderr, "%d: unexpected line inside book: \"%s\"\n",
		  lineNo, s);
	  goto finally;
	}
	break;
      case ctLevelItem:
	if (!strcmp(p, "level")) {
	  int levelNumber;
	  char *p2[2];
	  p2[0] = strtok(NULL, " \t\n\r");
	  p2[1] = strtok(NULL, "");
	  if(p2[1] ||
	     (levelNumber = atoi(p2[0])) < 0 ||
	     levelNumber > MAX_TEDAX_LEVELS) {
	    fprintf(stderr, "%d: incorrect level line: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }
	  levelItem[levelItems].level = levelNumber;
	} else if (!strcmp(p, "fullname")) {
	  char *p2[1];
	  p2[0] = strtok(NULL, "");
	  if(!p2[0]) {
	    fprintf(stderr, "%d: incorrect fullname line: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }
	  strcpy(levelItem[levelItems].fullName, p2[0]);
	} else if (!strcmp(p, "shortname")) {
	  char *p2[1];
	  p2[0] = strtok(NULL, "");
	  if(!p2[0]) {
	    fprintf(stderr, "%d: incorrect shortname line: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }
	  strcpy(levelItem[levelItems].shortName, p2[0]);
	} else if (!strcmp(p, "filename")) {
	  char *p2[1];
	  p2[0] = strtok(NULL, "");
	  if(!p2[0]) {
	    fprintf(stderr, "%d: incorrect filename line: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }
	  //	  strcpy(levelItem[levelItems].fileName, p2[0]);
	  if (!strcasecmp(p2[0], lastAudioFileName)) {
	    fprintf(stderr, "  cached %21s: %7ld milliseconds, currLev %2d\n",
		    lastAudioFileName, (long)lastAudioFileLenMS, currLevel);
	    //	    levelItem[levelItems].fileLenMS = lastAudioFileLenMS;
	  } else {
	    char *nameP = malloc(strlen(p2[0]+16));
	    FILE *fp = NULL;
	    struct AudioInfo ai = {0};
	    strcpy(nameP, p2[0]);
	    strcpy(lastAudioFileName, nameP);
	    lastAudioFileLenMS = -1;

	    if (!(fp = fopen(nameP, "rb"))) {
	      sprintf(nameP, "audio/%s", p2[0]);
	      if (!(fp = fopen(nameP, "rb"))) {
		fprintf(stderr, "%d: couldn't open \"%s\" nor \"%s\"\n",
			lineNo, p2[0], nameP);
		goto finally;
	      }
	    }
	    if (GetAudioInfoFP(fp, &ai, NULL, 0)) {
	      fprintf(stderr, "%d: couldn't analyze length of \"%s\"\n",
		      lineNo, nameP);
	      goto finally;
	    }
	    lastAudioFileLenMS = (s_int32)floor(ai.seconds*1000.0);
	    //	    levelItem[levelItems].fileLenMS = lastAudioFileLenMS;
	    fprintf(stderr, "  read %23s: %7ld milliseconds, currLev %2d\n",
		    nameP, (long)lastAudioFileLenMS, currLevel);
	    fclose(fp);
	    free(nameP);
	  }
	} else if (!strcmp(p, "filetime")) {
	  char *p2[1];
	  char *colon;
	  p2[0] = strtok(NULL, "");
	  if(!p2[0]) {
	    fprintf(stderr, "%d: incorrect filetime line: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }

	  colon = strchr(p2[0], ':');
	  if (colon) {
	    /* Time format: mm:ss[.hh] */
	    levelItem[levelItems].fileStartMS =
	      (int)(60000.0*strtod(p2[0], NULL) +
		    1000.0*strtod(colon+1, NULL));
	  } else {
	    /* Time format: ss[.hh] */
	    levelItem[levelItems].fileStartMS =
	      (int)(1000.0*strtod(p2[0], NULL));
	  }
#if 0
	  printf("### %16s %6d\n",
		 levelItem[levelItems].ident, levelItem[levelItems].fileMS);
#endif
	} else if (!strcmp(p, "text")) {
	  char *p2[1];
	  p2[0] = strtok(NULL, "");
	  if(!p2[0]) {
	    fprintf(stderr, "%d: incorrect text line: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }
	  strcpy(levelItem[levelItems].text, p2[0]);
	} else if (!strcmp(p, "end")) {
	  char *p2[2];
	  struct TedaxLevelItem *li = levelItem+levelItems;
	  p2[0] = strtok(NULL, " \t\n\r");
	  p2[1] = strtok(NULL, "");
	  if (p2[1] || !p2[0] || strcmp(p2[0], "levelitem")) {
	    fprintf(stderr, "%d: incorrect end line inside levelitem: \"%s\"\n",
		    lineNo, s);
	    goto finally;
	  }

	  /*
	    Write levelItem structure into binBuffer
	  */
	  if (li->level < 0 || li->level > currLevel+1 ||
	      li->level > MAX_TEDAX_LEVELS) {
	    fprintf(stderr, "%d: bad level inside levelitem: \"%s\", li->level %d, currLevel %d\n",
		    lineNo, s, li->level, currLevel);
	    goto finally;
	  }
	  strcpy(li->fileName, lastAudioFileName);
	  li->fileLenMS = lastAudioFileLenMS;
	  if (li->fileStartMS >= lastAudioFileLenMS) {
	    fprintf(stderr, "%d: file %s start point (%d ms) >= "
		    "file length (%d ms)\n",
		    lineNo, li->fileName, (int)li->fileStartMS, (int)lastAudioFileLenMS);
	    goto finally;
	  }
	  currLevel = li->level;
	  lastLevelItem[li->level] = &levelItem[levelItems];
	  if (currLevel > 0) {
	    lastLevelItem[li->level-1]->subLevelSize++;
	  }
	  itemsPerLevel[currLevel]++;
	  if (++levelItems > MAX_TEDAX_LEVEL_ITEMS) {
	    fprintf(stderr, "%d: too many levelItems (%d)\n",
		    lineNo, levelItems);
	    goto finally;
	  }
	  context = ctTedax;
	} else {
	  fprintf(stderr, "%d: unexpected line inside levelitem: \"%s\"\n",
		  lineNo, s);
	  goto finally;
	}
	break;
      default:
	fprintf(stderr, "%d: unexpected line: \"%s\"\n", lineNo, s);
	goto finally;
      }
    }
  }

  /* Only accept the file if we are in the correct context:
     not before tEDAx has been defined, and not inside a begin/end pair */
  if (context == ctTedax) errCode = 0;
 finally:
  return errCode;
}





void UpdateStringPointers(void) {
  int i;
#ifdef DEBUG
  printf("UpdateStringPointers:\n");
#endif
  for (i=0; i<pointersUsed; i++) {
    int a = GetSInt32(pointers[i]);
#ifdef DEBUG
    printf("  Pointer %x at %x: %x+%x = %x\n",
	   i, pointers[i], a, binBufferUsed, a+binBufferUsed);
#endif
    a += binBufferUsed;
    InsertSInt32(a, pointers[i]);
  }
}


int LevelItemCompare(const struct TedaxLevelItem *li1,
		     const struct TedaxLevelItem *li2) {
  return li1->level - li2->level;
}

int EmitBinary(FILE *fp) {
  int errCode = -1;

  /* Update book.levelItems field */
  InsertSInt16(levelItems, 6);

  /* Update string pointers from offsets to start of string buffer
     to actual file offset values. */
  UpdateStringPointers();

  /* Output binBuffer */
  if (fwrite(binBuffer, sizeof(char), binBufferUsed, fp) != binBufferUsed) {
    fprintf(stderr, "Couldn't write to binary buffer file\n");
    goto finally;
  }

  /* Output stringBuffer */
  if (fwrite(stringBuffer, sizeof(char), stringBufferUsed, fp) !=
      stringBufferUsed) {
    fprintf(stderr, "Couldn't write strings to file\n");
    goto finally;
  }

  errCode = 0;
 finally:
  return errCode;
}




int SortItems(void) {
  int errCode = -1;
  int i;
#if 0
  char lastFileName[MAX_TEDAX_LINE_LENGTH] = "";
  s_int32 lastFileLen;
#endif
  
  for (i=0; i<MAX_TEDAX_LEVELS+1; i++) {
    if (i < MAX_TEDAX_LEVELS) levelOffset[i+1] =
				  levelOffset[i]+itemsPerLevel[i];
    printf("Level %d has %4d items, offset %6d\n",
	   i, itemsPerLevel[i], levelOffset[i]);
  }

#if 0
  /*
    Fill in file names.
   */
  for (i=0; i<levelItems; i++) {
    struct TedaxLevelItem *li = levelItem+i;
    if (li->fileName[0]) {
      /* New file name: copy it and its length to memory. */
      strcpy(lastFileName, li->fileName);
      lastFileLen = li->fileLenMS;
    } else {
      /* No new file name: use the one and length from memory. */
      strcpy(li->fileName, lastFileName);
      li->fileLenMS = lastFileLen;
    }
  }
#endif
  
  /*
    If short name is available and long name is not,
    copy short name to long name.
   */
  for (i=0; i<levelItems; i++) {
    struct TedaxLevelItem *li = levelItem+i;
    if (!li->fullName[0] && li->shortName[0]) {
      strcpy(li->fullName, li->shortName);
    }
  }

  printf("Pre-sorting:\n");
  for (i=0; i<levelItems; i++) {
    struct TedaxLevelItem *li = levelItem+i;
    if (li->subLevelSize) {
      li->subLevelIdx = levelOffset[li->level+1];
      levelOffset[li->level+1] += li->subLevelSize;
    }
#if 0
    printf("i %4d: lev %d, ident %-16s pIdx %2d, sIdx %2d, sSize %2d %6d/%6d\n",
	   i, li->level, li->ident,
	   li->parentIdx, li->subLevelIdx, li->subLevelSize,
	   li->fileStartMS, li->fileLenMS);
#else
    printf("i %4d: lev %d, ident %-16s pIdx %2d, sIdx %2d, sSize %2d\n",
	   i, li->level, li->ident,
	   li->parentIdx, li->subLevelIdx, li->subLevelSize);
#endif
  }

  qsort(levelItem, levelItems, sizeof(levelItem[0]), (void *)LevelItemCompare);

  /* Update parent index for root */
  levelItem[0].parentIdx = 0xffff;

  printf("\nAfter sorting with indices set:\n");
  for (i=0; i<levelItems; i++) {
    int j;
    struct TedaxLevelItem *li = levelItem+i;

    /* Update parent indices */
    if (li->subLevelIdx) {
      for (j=0; j<li->subLevelSize; j++) {
	levelItem[li->subLevelIdx+j].parentIdx = i;
      }
    }

    AddUInt16(li->parentIdx);    /* VSDSP: levelItem[i].parentIdx */
    AddUInt16(li->subLevelIdx);  /* VSDSP: levelItem[i].subLevelIdx */
    AddUInt16(li->subLevelSize); /* VSDSP: levelItem[i].subLevelSize */
    AddUInt16(li->fileLenMS/1000);/* VSDSP: levelItem[i].fileLenMS */
    if (addIdentifiers) {
      AddString(li->ident);      /* VSDSP: levelItem[i].identP */
    } else {
                                 /* VSDSP: levelItem[i].identP */
      AddString("Identifier information not available");
    }
    AddString(li->fullName);     /* VSDSP: levelItem[i].fullNameP */
    AddString(li->shortName);    /* VSDSP: levelItem[i].shortNameP */
    AddString(li->text);         /* VSDSP: levelItem[i].textP */
    AddString(li->fileName);     /* VSDSP: levelItem[i].fileNameP */
    AddSInt32(li->fileStartMS);  /* VSDSP: levelItem[i].fileStartMS */
#if 0
    printf("i %4d: lev %d, ident %-16s pIdx %2d, sIdx %2d, sSize %2d %6d/%6d\n",
	   i, li->level, li->ident,
	   li->parentIdx, li->subLevelIdx, li->subLevelSize,
	   li->fileStartMS, li->fileLenMS);
#else
    printf("i %4d: lev %d, ident %-16s pIdx %2d, sIdx %2d, sSize %2d\n",
	   i, li->level, li->ident,
	   li->parentIdx, li->subLevelIdx, li->subLevelSize);
#endif
  }

  errCode = 0;
  // finally:
  return errCode;
}



int ProcessTedax(const char *inName, const char *outBase) {
  int errCode = -1;
  char s[MAX_TEDAX_LINE_LENGTH];
  FILE *inFp = NULL, *outFp = NULL;
  
  printf("inName \"%s\", outBase \"%s\"\n",
	 inName, outBase);

  /* Initialize structures. */
  binBufferUsed = 0;
  stringBufferUsed = 0;
  pointersUsed = 0;
  levelItems = 0;

  if (!(inFp = fopen(inName, "r"))) {
    fprintf(stderr, "Couldn't open read file %s\n", inName);
    goto finally;
  }

  if (ReadTedaxFile(inFp)) {
    fprintf(stderr, "Problem with reading file %s\n", inName);
    goto finally;
  }

  if (SortItems()) {
    fprintf(stderr, "Can't sort items\n");
    goto finally;
  }

  sprintf(s, "%s.tbf", outBase);
  if (!(outFp = fopen(s, "wb"))) {
    fprintf(stderr, "Couldn't open write file %s\n", s);
    goto finally;
  }
 
  if (EmitBinary(outFp)) {
    fprintf(stderr, "Problem with writing file %s\n", s);
    goto finally;
  }

  printf("%s: %ld bytes -> %s: %ld bytes\n",
	 inName, (long)ftell(inFp), s, (long)ftell(outFp));
  
  fclose(outFp);
  outFp = NULL;

  fclose(inFp);
  inFp = NULL;

 
  errCode = 0;
 finally:
  if (outFp) {
    fclose(outFp);
    outFp = NULL;
  }
  if (inFp) {
    fclose(inFp);
    inFp = NULL;
  }
  return errCode;
}




int main(int argc, char **argv) {
  int errCode = EXIT_FAILURE;
  int i;
  char binBase[MAX_TEDAX_LINE_LENGTH] = "";
  
  for (i=1; i<argc; i++) {
    if (!strcmp(argv[i], "-h")) {
      printf("tEDAxToVsdsp v" VERSION "\n");
      printf("\nUsage:\n%s [-h] [-s|-S] [-o outBase] [file1 [file2 [...]]]\n", argv[0]);
      printf("outBase\tOutput base name, must be defined before input fileX\n"
	     "fileX\tInput file name\n"
	     "-s|-S\tStrip|don't strip symbolic information\n"
	     "-h\tShow this help\n");
      errCode = EXIT_SUCCESS;
      goto finally;
    } else if (!strcmp(argv[i], "-s")) {
      addIdentifiers = 0;
    } else if (!strcmp(argv[i], "-S")) {
      addIdentifiers = 1;
    } else if (!strcmp(argv[i], "-o")) {
      if (i >= argc-1) {
	fprintf(stderr, "%s: No argument for parameter \"%s\"\n",
		argv[0], argv[i]);
	goto finally;
      }
      strcpy(binBase, argv[++i]);
    } else {
      /* If no basename has been defined, autogenerate one. */
      if (!binBase[0]) {
	char *tc;
	strcpy(binBase, argv[i]);
	/* If there is at least one .suffix in the basename, remove it.
	   E.g. "mytedax.txt" becomes "mytedax". */
	if ((tc = strrchr(binBase, '.'))) {
	  *tc = '\0';
	}
      }
      if (ProcessTedax(argv[i], binBase)) {
	fprintf(stderr, "%s: abort\n", argv[0]);
	goto finally;
      }
    }
  }
  
  errCode = EXIT_SUCCESS;
 finally:
  return errCode;
}
