/// \file main.c VS1010D VSOS Executable main file
/// This is a starting point for developing VS1010D DLX libraries and executables

#include <vo_stdio.h>
#include <volink.h>     // Linker directives like DLLENTRY
#include <apploader.h>  // RunLibraryFunction etc
#include <vs1010dRom.h>
#include <vo_gpio.h>
#include <vs1010c.h>
#include <playerinfo.h>
#include <string.h>
#include <stdlib.h>
#include <protocol.h>
#include <spitv.h>
#include <lcd.h>
#include <audio.h>
#include "book.h"

#define DEFAULT_VOLUME 48

#define VERSION 20251021

/*
  Define following if you want to use button input.
  If defined, be sure to connect to correct I/O pins.
  Can be defined together with ENABLE_UART_INPUT.
 */
#if 1
#  define ENABLE_BUTTON_INPUT
#endif

/*
  Define following if you want to use UART input.
  If defined, be sure to connect to correct I/O pins.
  Can be defined together with ENABLE_BUTTON_INPUT.
 */
#if 1
#  define ENABLE_UART_INPUT
#endif

#define FILE_STR_MAX_LEN 128

/* Reads the string, which filePos parameter points at, and stores it to fileStr parameter */
void GetFileString(register FILE *f, register s_int32 filePos, char *fileStr) {
  if (filePos) {
    s_int32 origPos = f->pos;
    int c;
    int n = 0;
    f->pos = filePos;
    /* Read and write to string at most FILE_STR_MAX_LEN-1 characters */
    while (++n < FILE_STR_MAX_LEN && (c = fgetc(f))) {
      *fileStr++ = c;
    }
    f->pos = origPos;
  }
  /* Always NUL-terminate string */
  *fileStr = '\0';
}


/* Prints the string, which filePos parameter points at */
void PrintFileString(register FILE *f, register s_int32 filePos) {
  if (filePos) {
    s_int32 origPos = f->pos;
    int c;
    f->pos = filePos;
    while ((c = fgetc(f)) > 0) {
      fputc(c, stdout);
    }
    f->pos = origPos;
  }
  return;
}



struct Book book;


/* Prints all available books in target folder, and returns the amount of them */
u_int16 PrintBooks(register FILE *f, char *mode, char *fileSpec, u_int16 cursor) {
  u_int16 totalBooks = 0;
  while (1) {
    sprintf(mode, "rb#%u", totalBooks);
    f = fopen(fileSpec, mode);
    if (f) {
      fread(&book, sizeof(book), 1, f);
      totalBooks++;
      PrintFileString(f, book.nameP);
      if (cursor == totalBooks) {
	printf(" <");
      }
      printf("\n");
      fclose(f);
    } else {
      printf("\n\nClose the program");
      if (cursor == totalBooks + 1) {
	printf(" <");
      }
      printf("\n");
      break;
    }
  }
  return totalBooks;
}


#ifdef ENABLE_BUTTON_INPUT
/* Checks for any button presses */
void ReadButtons(u_int16 *btn, char *press) {
     const int buttons[6] = {0x21, 0x22, 0x23, 0x24, 0x25, 0x26}; // Buttons' Gpio pins (MAKE SURE THESE ARE CORRECT)
     u_int16 i;
     u_int32 startTime, currTime;
     for (i = 0; i < sizeof(buttons); i++) {
	  if (GpioReadPin(buttons[i])) {
	       startTime = ReadTimeCount();
	       while (GpioReadPin(buttons[i])) {
		    currTime = ReadTimeCount() - startTime;
		    if (currTime > 1000) { // If button is held for atleast 500ms
			 *btn = i + 1;
			 *press = 'l';
			 return;
		    }
	       }
	       *btn = i + 1;
	       *press = 's';
	       return;
	  }
     }
     *btn = 0;
     return;
}
#endif



s_int32 startTime;
s_int32 endTime;
s_int32 skipTime;
u_int16 skip;
u_int16 next;
u_int16 toc; // Table Of Contents


/* Player callback function */
void MyPlayerCallback(AUDIO_DECODER *auDec, u_int16 samples) {
  static u_int16 btn = 0;
  static char press = ' ';
  static u_int16 lastPressed = 0;
  static char uartChar = ' ';
     
  if (startTime > 0 && player.auDec.cs.playTimeSeconds < startTime) {
    player.auDec.cs.goTo = (s_int16)startTime;
  } else if (player.auDec.cs.playTimeSeconds >= endTime && endTime > 0) {
    next = 1;
    player.auDec.cs.cancel = 1;
  }
     
#ifdef ENABLE_BUTTON_INPUT
  ReadButtons(&btn, &press);
#endif
#ifdef ENABLE_UART_INPUT
  if (!FIFOYEmpty(&uart0Fifo)) {
    uartChar = FIFOYGet(&uart0Fifo);
  } else {
    uartChar = ' ';
  }
#endif
     
  if (uartChar == 'p' || (btn == 1 && btn != lastPressed)) { // If 'p' was received via UART OR button 1 was pressed
    if (player.auDec.cs.playTimeSeconds - startTime < 3) {
      next = -1;
      player.auDec.cs.cancel = 1;
    } else {
      player.auDec.cs.goTo = (s_int16)startTime;
    }
  } else if (uartChar == 'n' || (btn == 2 && btn != lastPressed)) { // If 'n' was received via UART OR button 2 was pressed
    next = 1;
    player.auDec.cs.cancel = 1;
  } else if (uartChar == '1' || (btn == 3 && press == 'l' && btn != lastPressed)) { // If '1' was received via UART OR button 3 was long pressed
    printf("\nGoing back 60 seconds");
    if (player.auDec.cs.playTimeSeconds - 60 >= startTime) {
      player.auDec.cs.goTo = (s_int16)(player.auDec.cs.playTimeSeconds - 60);
    } else {
      skipTime = player.auDec.cs.playTimeSeconds - 60;
      skip = -1;
      player.auDec.cs.cancel = 1;
    }
  } else if (uartChar == '2' || (btn == 3 && press == 's' && btn != lastPressed)) { // If '2' was received via UART OR button 3 was short pressed
    printf("\nGoing back 10 seconds");
    if (player.auDec.cs.playTimeSeconds - 10 >= startTime) {
      player.auDec.cs.goTo = (s_int16)(player.auDec.cs.playTimeSeconds - 10);
    } else {
      skipTime = player.auDec.cs.playTimeSeconds - 10;
      skip = -1;
      player.auDec.cs.cancel = 1;
    }
  } else if (uartChar == '3' || (btn == 4 && press == 's' && btn != lastPressed)) { // If '3' was received via UART OR button 4 was short pressed
    printf("\nSkipping forward 10 seconds");
    if (player.auDec.cs.playTimeSeconds + 10 < endTime && endTime > 0) {
      player.auDec.cs.goTo = (s_int16)(player.auDec.cs.playTimeSeconds + 10);
    } else {
      skipTime = player.auDec.cs.playTimeSeconds + 10;
      skip = 1;
      player.auDec.cs.cancel = 1;
    }
  } else if (uartChar == '4' || (btn == 4 && press == 'l' && btn != lastPressed)) { // If '4' was received via UART OR button 4 was long pressed
    printf("\nSkipping forward 60 seconds");
    if (player.auDec.cs.playTimeSeconds + 60 < endTime && endTime > 0) {
      player.auDec.cs.goTo = (s_int16)(player.auDec.cs.playTimeSeconds + 60);
    } else {
      skipTime = player.auDec.cs.playTimeSeconds + 60;
      skip = 1;
      player.auDec.cs.cancel = 1;
    }
  } else if (uartChar == '-' || (btn == 5 && press == 's' && btn != lastPressed)) { // If '-' was received via UART OR button 5 was short pressed
    if (player.volume < 96) { // If volume isn't at min
      // Lower the volume by 6db
      player.volume += 12;
      printf("\nVolume lowered to -%ddB", player.volume/2);
    }
  } else if (uartChar == '+' || (btn == 6 && btn != lastPressed)) { // If '+' was received via UART OR button 6 was pressed
    if (player.volume > 0) { // If volume isn't at max
      // Raise the volume by 6dB
      player.volume -= 12;
      printf("\nVolume raised to -%ddB", player.volume/2);
    }
  } else if (uartChar == '<' || (btn == 5 && press == 'l' && btn != lastPressed)) { // If '<' was received via UART OR button 5 was long pressed
    printf("\nReturning to table of contents...");
    toc = 1;
    player.auDec.cs.cancel = 1;
  }
  lastPressed = btn;
}


struct LevelMem {
  s_int16 currIdx;
  s_int16 left;
};


s_int32 levelItemStartPos;
struct LevelItem levelItems[MAX_BOOK_LEVELS+1];
struct LevelMem levelMem[MAX_BOOK_LEVELS+1];

#if 0
void PrintLevelItems(FILE *f, int itemLevel) {
  struct LevelItem *li = levelItems;
  struct LevelMem *lm = levelMem;
  int i;
  
  for (i=0; i<=itemLevel; i++) {
    printf("L%d parent %x, subLevelIdx %x, subLevelSize %x, "
	   "str %lx %lx %lx\n  ",
	   i, li->parentIdx, li->subLevelIdx, li->subLevelSize,
	   li->identP, li->fullNameP, li->shortNameP);
    PrintFileString(f, li->fullNameP);
    printf("\n  text %lx, file %lx, start %lx\n",
	   li->textP, li->fileNameP, li->fileStartMS);
    printf("  lmCI %x, lmLeft %x\n", lm->currIdx, lm->left);
    lm++;
    li++;
  }
}
#endif

int GetLevelItem(FILE *f, struct LevelItem *li, s_int16 itemNumber) {
  fseek(f, levelItemStartPos+2*sizeof(struct LevelItem)*(s_int32)itemNumber,
	SEEK_SET);
#if 0
  printf("GetLevelItem: level %d, idx %2d, fseek(%04lx): slI %d, slS %d\n",
	 li-levelItems, itemNumber, f->pos, li->subLevelIdx, li->subLevelSize);
#endif
  if (fread(li, sizeof(levelItems[0]), 1, f) != 1) {
    printf("E: read levelItem\n");
    return -1;
  }
#if 0
  PrintLevelItems(f, li-levelItems);
#endif
  return 0;
}


enum states{library, contents, audioplayer, finish, error};


/* Prints all available books, and let's user pick which audiobook to play */
int Library(FILE **f, int *currLevel) {
  u_int16 lastPressed, btnPressed, totalBooks;
  u_int16 btn = 0;
  u_int16 cursor = 1;
  char press, uartChar;
  char mode[9];
  char fileSpec[128];
  strcpy(fileSpec, "S:/books/*.tbf");
  while (1) {
    printf("\n\nChoose a book:\n");
    totalBooks = PrintBooks(*f, mode, fileSpec, cursor); // Print all available books
#if 0
    printf("##1 totB %d\n", totalBooks);
#endif
    uartChar = ' ';
    btnPressed = 0;
    while (uartChar == ' ' && !btnPressed) { // Wait for user inputs
#ifdef ENABLE_BUTTON_INPUT
      ReadButtons(&btn, &press);
      if ((btn == 1 || btn == 2 || btn == 6) && btn != lastPressed) {
	btnPressed = 1;
      }
      lastPressed = btn;
#endif
#ifdef ENABLE_UART_INPUT
      if (!FIFOYEmpty(&uart0Fifo)) {
	uartChar = FIFOYGet(&uart0Fifo);
      }
#endif
    }
    if (uartChar == 'p' || btn == 1) { // If 'p' was received via UART OR button 1 was pressed
      if (cursor > 1) {
	cursor--;
      }
    } else if (uartChar == 'n' || btn == 2) { // If 'n' was received via UART OR button 2 was pressed
      if (cursor <= totalBooks) {
	cursor++;
      }
    } else if (uartChar == '>' || btn == 6) { // If '>' was received via UART OR button 6 was pressed
      if (cursor == totalBooks + 1) { // Close the program
	return finish;
      } else {
	break;
      }
    }
  }
  sprintf(mode, "rb#%u", cursor-1);
  *f = fopen(fileSpec, mode);
  if (!(*f)) {
    printf("E: open book\n");
    return error;
  }
  if (fread(&book, sizeof(book), 1, *f) != 1) {
    printf("E: read book\n");
    return error;
  }
  if (book.identifier != BOOK_IDENTIFIER) {
    printf("E: Not a book\n");
    return error;
  }
  levelItemStartPos = ftell(*f);
  *currLevel = 1;
  memset(levelMem, 0, sizeof(levelMem));
  GetLevelItem(*f, levelItems+0, 0);
  levelMem[1].currIdx = levelItems[0].subLevelIdx;
  levelMem[1].left = levelItems[0].subLevelSize - 1;
  return contents;
}


/* Prints book's table of contents and lets user pick which part to play */
int PrintTableOfContents(FILE *f, int *currLevel) {
  u_int16 i, j, lastPressed, btnPressed;
  u_int16 btn = 0;
  char press, uartChar;
  char fileStr[FILE_STR_MAX_LEN];
  
  while (1) {
    struct LevelItem *li = levelItems;
    struct LevelMem *lm = levelMem;
    GetFileString(f, book.levelNameP[*currLevel], fileStr);
    printf("\n\n\nChoose a %s:\n", fileStr);
#if 0
    printf("--------- cL %d\n", *currLevel);
    PrintLevelItems(f, *currLevel);
    printf("---------\n");
#endif
#if 0
    printf("#3 *currL %d\n", *currLevel);
#endif
    for (i = 0; i < *currLevel; i++) {
#if 0
      printf("#4 i %d\n", i);
#endif
      for (j = 1; j <= i; j++) {
	printf("-");
      }
      GetLevelItem(f, li, lm->currIdx);
      PrintFileString(f, li->fullNameP);
      li++; lm++;
      printf("\n");
    }
    for (i = li[-1].subLevelIdx; i < li[-1].subLevelSize + li[-1].subLevelIdx;
	 i++) {
#if 0
      printf("#5");
#endif
      for (j = 1; j <= *currLevel; j++) {
	printf("-");
      }
      GetLevelItem(f, li, i);
      PrintFileString(f, li->fullNameP);
      if (i == lm->currIdx) {
	printf(" <");
      }
      printf("\n");
    }
    uartChar = ' ';
    btnPressed = 0;
    while (uartChar == ' ' && !btnPressed) { // Wait for user inputs
#ifdef ENABLE_BUTTON_INPUT
      ReadButtons(&btn, &press);
      if ((btn == 1 || btn == 2 || btn == 5 || btn == 6) && btn != lastPressed) {
	btnPressed = 1;
      }
      lastPressed = btn;
#endif
#ifdef ENABLE_UART_INPUT
      if (!FIFOYEmpty(&uart0Fifo)) {
	uartChar = FIFOYGet(&uart0Fifo);
      }
#endif
    }
    if (uartChar == 'p' || btn == 1) { // If 'p' was received via UART OR button 1 was pressed
      if (lm->left < li[-1].subLevelSize - 1) {
	lm->currIdx--;
	lm->left++;
      }
    } else if (uartChar == 'n' || btn == 2) { // If 'n' was received via UART OR button 2 was pressed
      if (lm->left > 0) {
	lm->currIdx++;
	lm->left--;
      }
    } else if (uartChar == '<' || btn == 5) { // If '<' was received via UART OR button 5 was pressed
      if (*currLevel > 1) {
	(*currLevel)--;
	lm--;
	li--;
      } else {
	if (f) { // Close the current book
	  fclose(f);
	  f = NULL;
	}
	return library; // Return to pick a new book
      }
    } else if (uartChar == '>' || btn == 6) { // If '>' was received via UART OR button 6 was pressed
      GetLevelItem(f, li, lm->currIdx);
      if (li->subLevelIdx) {
#if 0
	printf("#6: *oLev -> %d, oIdx %d, left %d\n",
	       *currLevel,
	       lm->currIdx,
	       lm->left);
#endif
	(*currLevel)++;
	lm++;
	lm->currIdx = li->subLevelIdx;
	lm->left = li->subLevelSize - 1;
	li++;
#if 0
	printf("#7: *cLev -> %d, cIdx %d, left %d\n",
	       *currLevel,
	       lm->currIdx,
	       lm->left);
#endif
      } else {
	return audioplayer; // Continue to player
      }
    }
  }
}


/* Used by audioplayer for skipping forward x amount of seconds */
int FastForward(FILE *f, int *currLevel) {
  u_int16 i;
  while (skipTime > levelItems[*currLevel].fileLenS) {
    skipTime -= levelItems[*currLevel].fileLenS;
    levelMem[*currLevel].currIdx = levelItems[(*currLevel)-1].subLevelIdx + levelItems[(*currLevel)-1].subLevelSize - 1;
    levelMem[*currLevel].left = 0;
    for (i = *currLevel; !levelMem[i].left; i--) {
      if (i == 1) {
	printf("\n\nReached the end of the book\n\n");
	if (f) { // Close the current book
	  fclose(f);
	  f = NULL;
	}
	return 0;
      }
    }
    *currLevel = i;
    levelMem[*currLevel].currIdx++;
    levelMem[*currLevel].left--;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    while (levelItems[*currLevel].subLevelIdx) {
      (*currLevel)++;
      levelMem[*currLevel].currIdx = levelItems[(*currLevel)-1].subLevelIdx;
      levelMem[*currLevel].left = levelItems[(*currLevel)-1].subLevelSize - 1;
      GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    }
  }
  GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx+1);
  while (skipTime > levelItems[*currLevel].fileStartMS/1000 && levelMem[*currLevel].left) {
    levelMem[*currLevel].currIdx++;
    levelMem[*currLevel].left--;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx+1);
  }
  GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
  return 1;
}


/* Used by audioplayer for going back x amount of seconds */
void Rewind(FILE *f, int *currLevel) {
  u_int16 i;
  while (skipTime < 0) {
    levelMem[*currLevel].currIdx = levelItems[(*currLevel)-1].subLevelIdx;
    levelMem[*currLevel].left = levelItems[(*currLevel)-1].subLevelSize-1;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    for (i = *currLevel; levelMem[i].currIdx == levelItems[i-1].subLevelIdx; i--) {
      if (i == 1) {
	printf("\n\nReached the beginning of the book\n\n");
	GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
	return;
      }
    }
    *currLevel = i;
    levelMem[*currLevel].currIdx--;
    levelMem[*currLevel].left++;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    while (levelItems[*currLevel].subLevelIdx) {
      (*currLevel)++;
      levelMem[*currLevel].currIdx = levelItems[(*currLevel)-1].subLevelIdx + levelItems[(*currLevel)-1].subLevelSize - 1;
      levelMem[*currLevel].left = 0;
      GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    }
    skipTime += levelItems[*currLevel].fileLenS;
  }
  while (skipTime < levelItems[*currLevel].fileStartMS/1000 && levelMem[*currLevel].currIdx > levelItems[(*currLevel)-1].subLevelIdx) {
    levelMem[*currLevel].currIdx--;
    levelMem[*currLevel].left++;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
  }
  return;
}


/* Used by audioplayer for skipping to next verse */
int Next(FILE *f, int *currLevel) {
  u_int16 i;
  if (levelMem[*currLevel].left) {
    levelMem[*currLevel].currIdx++;
    levelMem[*currLevel].left--;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
  } else {
    for (i = *currLevel; !levelMem[i].left; i--) {
      if (i == 1) {
	printf("\n\nReached the end of the book\n\n");
	if (f) { // Close the current book
	  fclose(f);
	  f = NULL;
	}
	return 0;
      }
    }
    *currLevel = i;
    levelMem[*currLevel].currIdx++;
    levelMem[*currLevel].left--;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    while (levelItems[*currLevel].subLevelIdx) {
      (*currLevel)++;
      levelMem[*currLevel].currIdx = levelItems[(*currLevel)-1].subLevelIdx;
      levelMem[*currLevel].left = levelItems[(*currLevel)-1].subLevelSize - 1;
      GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    }
  }
  return 1;
}


/* Used by audioplayer for skipping to previous verse */
void Previous(FILE *f, int *currLevel) {
  u_int16 i;
  if (levelMem[*currLevel].currIdx > levelItems[(*currLevel)-1].subLevelIdx) {
    levelMem[*currLevel].currIdx--;
    levelMem[*currLevel].left++;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
  } else {
    for (i = *currLevel; levelMem[i].currIdx == levelItems[i-1].subLevelIdx; i--) {
      if (i == 1) {
	printf("\n\nReached the beginning of the book\n\n");
	GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
	return;
      }
    }
    *currLevel = i;
    levelMem[*currLevel].currIdx--;
    levelMem[*currLevel].left++;
    GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    while (levelItems[*currLevel].subLevelIdx) {
      (*currLevel)++;
      levelMem[*currLevel].currIdx = levelItems[(*currLevel)-1].subLevelIdx + levelItems[(*currLevel)-1].subLevelSize - 1;
      levelMem[*currLevel].left = 0;
      GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx);
    }
  }
  return;
}


/* Prints the current verse, sets correct start and end times, and starts playing correct mp3 file */
int AudioPlayer(FILE *f, int *currLevel) {
  u_int16 i;
  char fileStr[FILE_STR_MAX_LEN];
  while (1) {
    GetFileString(f, levelItems[*currLevel].fileNameP, fileStr);
    printf("\n\n\n");
    // Print current part
    for (i = 1; i <= *currLevel; i++) {
      if (i == 2) {
	printf(": ");
      } else if (i == 3) {
	printf(" ");
      } else if (i == 4) {
	printf(":");
      }
      if (i == 3) {
	PrintFileString(f, levelItems[i].shortNameP);
      } else {
	PrintFileString(f, levelItems[i].fullNameP);
      }
    }
    printf("\n");
    PrintFileString(f, levelItems[*currLevel].textP);
    printf("\n\n\n\n");
    
    // Set start time
    if (skip) {
      startTime = skipTime;
      skip = 0;
    } else if (levelMem[*currLevel].currIdx == levelItems[(*currLevel)-1].subLevelIdx) {
      startTime = 0;
    } else {
      startTime = levelItems[*currLevel].fileStartMS / 1000;
    }
    // Set end time
    if (levelMem[*currLevel].left) {
      GetLevelItem(f, levelItems+(*currLevel), levelMem[*currLevel].currIdx + 1);
      endTime = levelItems[*currLevel].fileStartMS / 1000;
    } else {
      endTime = 0;
    }
    //printf("Start time: %ld, End time: %ld, Skip time: %ld, File lenght: %u\n", startTime, endTime, skipTime, levelItems[*currLevel].fileLenS);
    
    strcpy(player.fileSpec, "S:/audio/"); // Mp3 files' target directory
    strcat(player.fileSpec, fileStr); // File name
    PlayerPlayFile(); // Play mp3 file
    
    if (toc) { // If user wants to return to the table of contents
      toc = 0;
      return contents;
    } else if (skip == 1) { // If user wants to skip forwards x amount of seconds
      if (FastForward(f, currLevel)) {
	return audioplayer;
      } else { // If user reaches the end of the book --> return to library
	return library;
      }
    } else if (skip == -1) { // If user wants to go back x amount of seconds
      Rewind(f, currLevel);
      return audioplayer;
    } else if (next == 1) { // If user wants to go to next verse
      if (Next(f, currLevel)) {
	return audioplayer;
      } else { // If user reaches the end of the book --> return to library
	return library;
      }
    } else if (next == -1) { // If user wants to go to previous verse
      Previous(f, currLevel);
      return audioplayer;
    }
    return audioplayer;
  }
}


ioresult main(char *params) {
  ioresult retCode = S_ERROR;
  FILE *f;
  int currLevel;
  int state = 0;
  skip = 0;
  player.nextStep = 0;
  player.currentFile = 0;
  player.volume = DEFAULT_VOLUME;

  printf("Audio Book v%ld\n", VERSION);
  
  SetJmpiHookFunction(PlayerCallback, MyPlayerCallback); // Use MyPlayerCallback instead of the DefaultPlayerCallback

  while (1) {
    if (state == library) {
      state = Library(&f, &currLevel);
    } else if (state == contents) {
      state = PrintTableOfContents(f, &currLevel);
    } else if (state == audioplayer) {
      state = AudioPlayer(f, &currLevel);
    } else if (state == finish) {
      printf("\n\nClosing the program...\n\n");
      retCode = S_OK;
      break;
	  } else if (state == error) {
      break;
    }
  }
     
finally:
  if (f) {
    fclose(f);
    f = NULL;
  }
  SetJmpiHookFunction(PlayerCallback, DefaultPlayerCallback); // Restore the original player callback
  return retCode; 
}
