/// \file main.c VS1010D VSOS Executable main file
/// This is a starting point for developing VS1010D DLX libraries and executables

#include <vo_stdio.h>
#include <volink.h>     // Linker directives like DLLENTRY
#include <apploader.h>  // RunLibraryFunction etc
#include <vs1010dRom.h>
#include <vo_gpio.h>
#include <vs1010c.h>
#include <playerinfo.h>
#include <string.h>
#include <protocol.h>
#include <spitv.h>
#include <lcd.h>
#include <xvm.h>
#include <audio.h>
#include <usblowlib.h>
#include "src/usb_host/UsbFlash.h"
#include <swap.h>

void SetCoreClockVCO(register u_int32 vco_sdm_value);

struct {
	s_int16 core;
	s_int16 io;
	s_int16 analog;
} regulator = {CORE_1V9, IO_3V3, ANA_3V6};

UsbMscDeviceInfo myud = {0,0,{
	USB_DEVICE_ENDPOINT(0,0),HOST_TX_PID(PID_DATA0),64,0,
	USB_DEVICE_ENDPOINT(0,0),HOST_TX_PID(PID_DATA0),64,0,
	USB_DEVICE_ENDPOINT(0,0),HOST_TX_PID(PID_DATA0),64,0},0};

UsbMscDeviceInfo udDefaults = {0,0,{
	USB_DEVICE_ENDPOINT(0,0),HOST_TX_PID(PID_DATA0),64,0,
	USB_DEVICE_ENDPOINT(0,0),HOST_TX_PID(PID_DATA0),64,0,
	USB_DEVICE_ENDPOINT(0,0),HOST_TX_PID(PID_DATA0),64,0},0};

UsbMscDeviceInfo *udp = &myud;

u_int16 pktBuf[256];


#define E_NOTHING_DETECTED -2
#define E_NO_RESPONSE -3
#define E_STALLED -4
#define UTMI_AUTOSOF_ON() PERIP(USB_UTMIW) |= USB_UTMIW_AUTOSOF;
#define UTMI_AUTOSOF_OFF() PERIP(USB_UTMIW) &= ~USB_UTMIW_AUTOSOF;
#define GET_SOF_CNT() (PERIP(USB_UTMIR)&0x3FFF)
#ifndef PID_NAK
#define PID_NAK PID_NACK
#endif


u_int16 usbTimeoutTicks = 200;
#define START_TIMEOUT() u_int32 usbTime1 = ReadTimeCount()	
#define CHECK_TIMEOUT ((ReadTimeCount()-usbTime1) > usbTimeoutTicks)


u_int16 getDeviceDescriptor[] = {0x8006, 0x0001, 0x0000, 0x4000};
u_int16 getConfigurationDescriptor[] = {0x8006,0x0002,0x0000,0xff00};
u_int16 setAddress1[] = {0x0005,0x0100,0x0000,0x0000};
u_int16 setConfiguration1[] = {0x0009,0x0100,0x0000,0x0000};
u_int16 getMaxLun[] = {0xa1fe,0x0000,0x0000,0x0100};


/// Wait for condition of the USB peripheral (packet sent or received etc)
u_int16 UsbWaitFor(register u_int16 mask) {
	START_TIMEOUT();	
	u_int16 result;
	do {
		result = PERIP(USB_STATUS) & mask;
		if (CHECK_TIMEOUT) {
			printf("T1");
			return USB_ST_TIME;
		}
	} while (!result); // && timeout--);
	PERIP(USB_ST) = result;
	return result;
}



void UsbSendToken(register EpInfo *ep, register u_int16 pidShifted) {
	static const VCODE vcode1[] = {
		WriteP3(USB_WRPTR, 0),
		WriteP3(USB_RDPTR, 0),
		WriteP3(USB_ST, USB_ST_RX | USB_ST_TIME | USB_ST_TX_EMPTY | USB_ST_MTERR),
		WriteP3(USB_EP_SEND0, USB_EP_SEND_TXR | (0 << USB_EP_SEND_ADDR_B) | 0), // for the token packet, the amount of data is fixed and 0 is put to the field
		0
	};
	
	if (!(PERIP(USB_CF) & USB_CF_USBENA)) return; //USB must be enabled, otherwise this code will deadlock
	while (GET_SOF_CNT() < 1000); //Need at least this much time remaining in the USB frame (750 seems to be the absolute minimum)
	
	USEY(USB_SEND_MEM + (0 * 64)) = ep->addr_ep; //token packet to slot 0
	Xvm((void*)vcode1);
	if (pidShifted == HOST_TX_PID(PID_SETUP)) { // Are we doing a SETUP, then reset data toggles
		ep->host_send_with_toggle = HOST_TX_PID(PID_DATA0);
	}
	PERIP(USB_ST) |= USB_STF_SOF;
	PERIP(USB_HOST) = pidShifted; //Start sending the PID opcode and parameter byte (address and endpoint)
	//The caller must ensure 3 us delay before pushing out the next packet
}
	


ioresult UsbSendData(register EpInfo *ep, register u_int16 pidShifted, register s_int16 bytes, register __i0 u_int16 *dataptr) {
	START_TIMEOUT();
	u_int16 r; 	
	do {
		s_int16 bytesToSend = bytes;
		if (bytesToSend > ep->size) bytesToSend = ep->size;
		if (CHECK_TIMEOUT) return E_NO_RESPONSE;
		
		// Start USB transaction by sending out a token (OUT or SETUP for outgoing data, IN for incoming data)		
		UsbSendToken(ep, pidShifted); //Send OUT or SETUP token
		//Prepare data packet in memory into slot 1
		memcpyXY((__y*)(USB_SEND_MEM + (1 * 64)), dataptr, 32);
		DelayL(5); // Delay amount is critical. Need to wait exact amount of 3 microseconds for the PID packet to clear out to the bus
		// Then send the data...
		PERIP(USB_ST) = USB_ST_RX | USB_ST_TIME | USB_ST_TX_EMPTY | USB_ST_MTERR; //first clear any pending flags
		PERIP(USB_EP_SEND0) = USB_EP_SEND_TXR | (1 << USB_EP_SEND_ADDR_B) | bytesToSend; //setup packet memory
		PERIP(USB_HOST) = ep->host_send_with_toggle; //start shifting out the data, using DATA0 or DATA1
		// Then wait for a response (ACK, NAK, STALL, TIMEOUT, SOF etc)
		
		r = UsbWaitFor(USB_ST_RX | USB_ST_SOF | USB_ST_TIME);
		if (r == USB_ST_RX) { //Response packet received
			u_int16 pid = PERIP(USB_ST) & 0xf;
			if (pid == PID_STALL) {
				ep->stalled = 1;
				return E_STALLED; //The other side will never be able to support this request
			} else if (pid == PID_ACK) {
				ep->host_send_with_toggle ^= (1<<15); //XOR the difference between DATA0 and DATA1
				if (!bytesToSend) { // we just sent out a zero length packet, it must be the last packet
					return S_OK;
				}
				bytes -= bytesToSend;
				dataptr += bytesToSend / 2;
				if ((bytes == 0) && (bytesToSend != 64)) return S_OK; //no ZLP needed
			}
		} 

	} while (bytes >= 0);
	return S_OK;
}
	



u_int16 usbTotalBytesReceived = 0;
u_int16 stickResetNeeded = 0;

int UsbReceiveData(register EpInfo *ep, register s_int16 bytes, register __i0 u_int16 *dataptr) {
	START_TIMEOUT();
	usbTotalBytesReceived = 0;
	memset(dataptr, 0, bytes/2); // I think it's nice to clear the receive buffer so that no code can ever parse data that was received for another request

	while (bytes){
		if (CHECK_TIMEOUT) return  E_NO_RESPONSE;

		// Start USB transaction by sending out a token (OUT or SETUP for outgoing data, IN for incoming data)
		UsbSendToken(ep, HOST_TX_PID(PID_IN)); //Send IN token
		
		// Then wait for a response (DATA, STALL, TIMEOUT, SOF etc)
		if ((UsbWaitFor(USB_ST_RX | USB_ST_SOF | USB_ST_TIME) == USB_ST_RX) //Response packet received
		&& ((PERIP(USB_ST) & 0x7) == 3)) { //DATA0 or DATA1 packet has been received
			u_int16 bytesReceived = (USEY(USB_RECV_MEM) & 0x03FF);
			u_int16 lengthWords = (bytesReceived + 1) >> 1;
			usbTotalBytesReceived += bytesReceived;
			bytes -= bytesReceived;
			if (bytes < 0) {stickResetNeeded = 1; return S_ERROR;} //too much data received
			memcpyYX(dataptr,(__y *)(USB_RECV_MEM+1),lengthWords);
			dataptr += lengthWords;
			if (bytesReceived < ep->size) return usbTotalBytesReceived;
		} 
		if ((PERIP(USB_ST) & 0xf) == PID_STALL) {
			ep->stalled = 1;
			return E_STALLED;
		}
	}
	return usbTotalBytesReceived;	
}




int UsbControlTransfer(u_int16 *setup, int lengthAndDirection, u_int16 *data) {
	u_int16 bytesIn = 0;
	UsbSendData(&udp->ep[0], HOST_TX_PID(PID_SETUP), 8, setup);

	// Control Input (get descriptor etc)
	if (lengthAndDirection > 0) {
		bytesIn = UsbReceiveData(&udp->ep[0], lengthAndDirection, data);
		if (bytesIn >= 0) UsbSendData(&udp->ep[0], HOST_TX_PID(PID_OUT), 0, NULL); // Send zero length packet
		return bytesIn;
	}

	// Control Output (set descriptor etc, rarely needed)	
	if (lengthAndDirection < 0) {
		UsbSendData(&udp->ep[0], HOST_TX_PID(PID_OUT), -lengthAndDirection, data);
	}
		
	// Receive zero length packet
	return UsbReceiveData(&udp->ep[0], 64, pktBuf); //GetZLP
}



const VCODE resetUsbHostHw[] = {
	SetBitsP3(ANA_CF3, ANA_CF3_UTMBIAS | ANA_CF3_UTM_ENA | ANA_CF3_480_ENA),
	WriteP3(USB_UTMIW, USB_UTMIW_ORIDE),
	WriteP3(USB_CF,	USB_CF_RST),
	WriteP3(USB_WRPTR, 0x0000),
	WriteP3(USB_RDPTR, 0x0000),
	ClearBitsP3(ANA_CF3, ANA_CF3_480_ENA),
	WriteP3(USB_CF, USB_CF_MASTER | USB_CF_USBENA),
	WriteP3(USB_UTMIW, USB_UTMIW_RCVSEL | USB_UTMIW_TERMSEL),
	0
};


ioresult UsbHostReset() {
	ioresult r;
	START_TIMEOUT();
	
	u_int32 time1 = ReadTimeCount();	
	
	memcpy(udp, &udDefaults, sizeof(udDefaults));
	{
		u_int16 i;
		for (i=0xfc80u; i<=0xfc9f; i++) {
			PERIP(i) = 0; //Clear ALL USB registers
		}
	}
	SetCoreClockVCO(0xFF87FFFF);
	Xvm((void*)resetUsbHostHw); // Run USB host reset sequence as vcode script to save instruction memory

	while ((PERIP(USB_UTMIR) & USB_UTMIR_LSTATE) == 0) { // Wait for USB device pull-up, our on-chip pulldown is active (host)
		if (CHECK_TIMEOUT) {
			PERIP(USB_UTMIW) = 0;
			PERIP(USB_CF) = 0; //Stop USB hardware
			return E_NOTHING_DETECTED;			
		}
	}
	
	DelayL(1000);
	PERIP(USB_UTMIW) = USB_UTMIW_ORIDE; // Set Reset state
	DelayL(100000);
	PERIP(USB_UTMIW) = USB_UTMIW_ORIDE | USB_UTMIW_RCVSEL | USB_UTMIW_TERMSEL | USB_UTMIW_AUTOSOF;	// Set normal state
	DelayL(100000);
		
	return S_OK;
}


// Decriptor types
#define T_DEV_DESCRIPTOR   1
#define T_CONF_DESCRIPTOR  2
#define T_IFACE_DESCRIPTOR 4
#define T_EP_DESCRIPTOR    5
#define T_STATUS_DESC      6

#define MASS_STORAGE_CLASS 0x08
#define SCSI_CMD_SET 0x06
#define BULK_ONLY_TRANSPORT 0x50
#define EP_0   0
#define EP_IN  1
#define EP_OUT 2 
#define ENDPOINT_TYPE_BULK 2
#define TWO_ENDPOINTS 2


register s_int16 ParseDescriptorPiece() { //, u_int16 *pkt) {
	static int nowParsingMscInterface = 0;
	u_int16 length = pktBuf[0] >> 8;
	u_int16 dtype = pktBuf[0] & 0xff;
				
	if (dtype == T_IFACE_DESCRIPTOR) { // Check if this interface is a mass storage interface with which we are compatible
		nowParsingMscInterface = (*((u_int32*)(&pktBuf[2])) == MASS_STORAGE_CLASS | (TWO_ENDPOINTS << 8) | (BULK_ONLY_TRANSPORT << 16) | (SCSI_CMD_SET << 24));
	} // ...the feeling when you can combine four separate byte checks into one 32-bit comparison! :D

	if (dtype == T_EP_DESCRIPTOR && nowParsingMscInterface) {
		u_int16 epat = pktBuf[1];
		if ((epat & 0xF0FF) == (0x8000 | ENDPOINT_TYPE_BULK)) { // BULK IN
			udp->ep[EP_IN].addr_ep = udp->ep[0].addr_ep | ((epat >> 8) & 0xf);
			udp->ep[EP_IN].size = (pktBuf[2] >> 8);
		}
		if ((epat & 0xF0FF) == (0x0000 | ENDPOINT_TYPE_BULK)) { // BULK OUT
			udp->ep[EP_OUT].addr_ep = udp->ep[0].addr_ep | ((epat >> 8));
			udp->ep[EP_OUT].size = (pktBuf[2] >> 8);			
		}
	}
	return length;
}

	
	
void ParseConfigurationDescriptor(register u_int16 len) { //, register u_int16 *pkt) {
	while (len) {
		u_int16 thisLen = ParseDescriptorPiece();
		len -= thisLen;
		MemCopyPackedBigEndian(pktBuf, 0, pktBuf, thisLen, len);
	}
	return;
}



ioresult UsbDeviceProbe() {
	u_int16 attempts;
	int r;
	usbTimeoutTicks = 500;
	Xvm((void*)resetUsbHostHw); // Set initial state
	DelayL(10000);
	if ((PERIP(USB_UTMIR) & USB_UTMIR_LSTATE) == 0) return E_NOTHING_DETECTED;
	for (attempts = 0; attempts < 10; attempts++) {
		UsbHostReset();
		DelayL(100000);
		if (UsbControlTransfer(getDeviceDescriptor, 18, &udp->deviceDescriptor) == 18) return S_OK;
		DelayL(10000);
	}	
	PERIP(USB_CF) = 0;
	return E_NOTHING_DETECTED;
}


#include "src/usb_host/scsihost.c"

extern const FILESYSTEM FatFileSystem;

ioresult usb_host_main() {
	int attempts = 0;
	ioresult r;
	PowerSetVoltages((void*)(&regulator));
	VODEV('U') = &usbFlash;
			
	while (lastReceivedCharUart0 != 27) {		
		printf("Please insert USB flash..\n");		
		while (UsbDeviceProbe()) {
			DelayL(10000);
		}

		printf("VendorID:%04X ProductID:%04X\n",Swap16(udp->vid),Swap16(udp->pid));
		printf("Scan and init filesystem...\n");
		
		if (ioctl(&usbFlash, IOCTL_RESTART, NULL) == (IOCTL_RESULT)&FatFileSystem) {
			printf("Device '%s' has FAT filesystem, good. Playing *.MP3 from root..\n", Identify(VODEV('U')));
			shellcmd("play","U:*.MP3");
		} else {
			printf("Init error at ioctl_restart\n");
		}
		
		printf("Done, please remove USB flash..\n");
		
		usbTotalSectors32 = 0;		
		Xvm((void*)resetUsbHostHw); // Set initial state				
		while ((PERIP(USB_UTMIR) & USB_UTMIR_LSTATE) != 0) {
			//Wait for no stick
		}

		printf("USB device disconnected\n");
	}
	VODEV('U') = NULL;
	return S_OK;
}

