#include <string.h> //memset
#include <vsos.h> //ioctl
#include <aucommon.h> // IOCTL audio defines
#include <vo_stdio.h>
#include "noisegate.h"

// Setup values from main()
u_int16 noisegateDBThreshold = 0;
u_int16 noisegateAttackUs = 100;
u_int16 noisegateHoldMs = 100;
u_int16 noisegateReleaseMs = 20;
// Processing values.
u_int16 noisegateAttackSpeed;
u_int16 noisegateReleaseSpeed;
u_int32 noisegateThreshold; //Mean square value
u_int32 noisegateHold;
// Contextual variables
u_int16 noisegateMultiplier; //Fader multiplier.

#define NG_FULL_OPEN 65535
#define NG_FULL_CLOSED 0
/* Fades only 16 MSb's.
   If samples are 32-bit, this leaves 16 LSb's untouched.
   While not entirely correct from a signal processing standpoint,
   pseudo-random noise in the 16 LSb's during fading are insignificant. */
void FadeOut(register s_int16 *d, register u_int16 samples) {
  u_int16 mult = noisegateMultiplier;
  s_int16 i;
  d += wordsPerSample-1;
  for (i=0; i<samples; i++) {
    *d = (s_int16)(((s_int32)(*d) * (s_int32)mult) >> 16);
    d += wordsPerSample;
    mult -= noisegateReleaseSpeed;
    if(mult > noisegateMultiplier){
      // Underflow. Zero one sample too early and
      // the rest and then return.
      noisegateMultiplier = NG_FULL_CLOSED;
      memset(d, 0, (samples - i) * wordsPerSample);
      return;
    }
    noisegateMultiplier = mult;
  }
}

/* Fades only 16 MSb's.
   If samples are 32-bit, this leaves 16 LSb's untouched.
   While not entirely correct from a signal processing standpoint,
   pseudo-random noise in the 16 LSb's during fading are insignificant. */
void FadeIn(register s_int16 *d, register u_int16 samples) {
  s_int16 i;
  d += wordsPerSample-1;
  for (i=0; i<samples; i++) {
    *d = (s_int16)(((s_int32)(*d) * (s_int32)noisegateMultiplier) >> 16);
    d += wordsPerSample;
    if (noisegateMultiplier >= NG_FULL_OPEN - noisegateAttackSpeed){
      noisegateMultiplier = NG_FULL_OPEN;
      return;
    }
    noisegateMultiplier += noisegateAttackSpeed;
  }
}
/* Calculates only 16MSb's
   If samples are 32-bit, this doesn't count 16 LSb's
   Those would only provide useless accuracy and make
   calculation harder. Returns mean square value. */
u_int32 CalcPower(const s_int16 *p, u_int16 samples) {
  u_int32 sumLo=0, sumHi=0, oldSumLo=0;
  s_int16 i;
  p += wordsPerSample -1;
  for (i=0; i<samples; i++) {
    sumLo += (s_int32)*p * (s_int32)*p;
    p += wordsPerSample;
    if (sumLo < oldSumLo) {
      sumHi++;
    }
    oldSumLo = sumLo;
  }
  // When bigger scale (not resolution) is needed,
  // use floating point numbers and beware of rounding errors!
  return (u_int32)((sumLo + sumHi*4294967296.0)/samples);
}

#define GATE_CLOSED 0
#define GATE_OPENING 1
#define GATE_OPEN 2
#define GATE_CLOSING 3

/* When NoisegateProcess gets signal above gateThreshold,
   opening is started with FadeIn().  After gate is open,
   level is monitored and if it goes below threshold,
   hold counter is started. When counter expires FadeOut()
   is called. If signal goes above threshold level, closing
   is stopped and opening is started again.*/
void NoisegateProcess(register s_int16 *p, register u_int16 samples){
  static u_int16 state = GATE_CLOSED;
  static u_int32 holdCounter = 0;
  u_int32 power;
  // Everything is above zero so noise gate disabled or no samples.
  if (!noisegateThreshold||!samples){
    return;
  }
  power = CalcPower(p,samples);

  switch(state){
  case GATE_CLOSED:
    if (power < noisegateThreshold){
      memset(p, 0, samples * wordsPerSample);
      break;
    }
    state = GATE_OPENING;
    // Intentional fall-through to opening.
  case GATE_OPENING:
    FadeIn(p,samples);
    if (noisegateMultiplier == NG_FULL_OPEN){
      state = GATE_OPEN;
    }
    break;
  case GATE_OPEN:
    if( power > noisegateThreshold){
      break;
    }
    state = GATE_CLOSING;
    holdCounter = noisegateHold;
    // Intentional fall-through to closing
  case GATE_CLOSING:
    if (power > noisegateThreshold){
      state = GATE_OPENING;
      // Fade in now and let next round put the gate open state.
      FadeIn(p,samples);
      break;
    }
    // Wait for hold time
    if(holdCounter > samples){
      holdCounter -= samples;
      break;
    }
    // Fade out
    FadeOut(p + (u_int16)holdCounter*wordsPerSample,
            samples - (u_int16)holdCounter);
    holdCounter = 0;

    if (noisegateMultiplier == NG_FULL_CLOSED){
      state = GATE_CLOSED;
    }
    break;
  default:
    // This should never run.
    state = GATE_CLOSED;
  }
}

void NoisegateSetParameters(register u_int16 thresholdDb,
                            register u_int16 attackUs, 
                            register u_int16 holdMs,
                            register u_int16 releaseMs){
  u_int32 tmp, sampleRate;
  // Comparison is done with RMS values of the signal,
  // without the square root.
  noisegateThreshold = DBToLin(thresholdDb);
  noisegateThreshold = noisegateThreshold * noisegateThreshold;
  //Get the sample rate.
  if (ioctl(stdaudioin, IOCTL_AUDIO_GET_IRATE, (char *)(&sampleRate))) {
    printf("Couldn't get samplerate\n");
    return;
  }
  // Get bits.
  wordsPerSample = ioctl(stdaudioin, IOCTL_AUDIO_GET_BITS, NULL)>>4;
  //Calculate speeds.
  tmp = (sampleRate * attackUs / 1000000) + 1;
  // Not exact, but doesn't divide by zero in any case.
  noisegateAttackSpeed = (u_int16)(65535 / tmp);
  noisegateHold = sampleRate/1000 * holdMs;
  tmp = sampleRate / 1000 * releaseMs + 1;
  tmp &= 0xffff;
  noisegateReleaseSpeed = (u_int16)(65535 / tmp);
  noisegateDBThreshold = thresholdDb;
  noisegateAttackUs = attackUs;
  noisegateHoldMs = holdMs;
  noisegateReleaseMs = releaseMs;
}
